#!/bin/bash

declare -a args=("${@}")

SCRIPT=$(readlink -f "$0"); PID=$$
DIR=$(cd "$(dirname $SCRIPT)" && pwd)
HOMEDIR=$(getent passwd pi | cut -d: -f6)

temp_dir="$DIR/tmp"
log_file="$DIR/.logfile"
roms_dir="$HOMEDIR/RetroPie/roms/mame-libretro"
assets_path="$HOMEDIR/ADMIN/MENU/.assets"
cafca_dir="$HOMEDIR/CAFCA/data/mame-libretro"
joytyper_dir="$HOMEDIR/ADMIN/.sys/joytyper"
download_dir="$HOMEDIR/downloads/MAME2003_Reference_Set_MAME0.78_ROMs_CHDs_Samples/roms"
torrent_file="$HOMEDIR/ADMIN/FILES/TORRENTS/mame078_romset.torrent"

prompt=$(sed -n '/^editor_prompt=/ {s/\(.*\)=//g;s/#.*$//;p}' "$assets_path/.settings")
preserve_roms=$(sed -n '/^editor_preserve_roms=/ {s/\(.*\)=//g;s/#.*$//;p}' "$assets_path/.settings")
t_stored="$(( $(cat /sys/module/kernel/parameters/consoleblank) / 60 ))"

finished=0; progress=0; total=0; p_prnt=0; p_max=1
play_sound=1

#________________________________________________________________________________

# SETUP

export NEWT_COLORS="$(cat $HOMEDIR/ADMIN/MENU/.assets/config/whiptail_theme.cfg)"

shopt -s expand_aliases
alias log_append="sudo tee -a \"$log_file\""
alias log_append_silent="sudo tee -a \"$log_file\" &>/dev/null"

sudo -E -u pi bash -c "dialog --create-rc /home/pi/.dialogrc; cp -f /home/pi/ADMIN/MENU/.assets/config/.dialogrc /home/pi/.dialogrc"
sudo rm $download_dir/*.* 2>/dev/null

[ -d $temp_dir ] && sudo rm -R $temp_dir; mktemp -d --suffix=_TEMP --tmpdir=$DIR &>/dev/null && { mv -f $DIR/tmp*_TEMP $temp_dir; for tmp in $temp_dir; do sudo chmod -R 775 $temp_dir; sudo chown -R pi:root $temp_dir; done; }
[[ $TERM == linux ]] && echo -en "\e]P700FC00"
echo -en "\033[9;0]" > /dev/tty1;

printf '' > "$log_file"
#printf '' > "$temp_dir/downloads"

clear

# -------------------------------------------------------------------------------

# MAIN

Main() {
  clear

  changed="$([ -f $temp_dir/.list ] && echo 1 || echo 0)"

  choice=$( \
    whiptail --menu "" --title "EDITOR" 0 0 0 \
      "ADD" "" \
      "EDIT" "" \
      "REFRESH" "$([[ $changed -eq 1 ]] && echo ' *')" \
      "REMOVE" "" \
      "RESTORE" "" \
      "APPLY" "$([[ $changed -eq 1 ]] && echo ' *')" \
    2>&1 1>$(tty))

  [[ $? -gt 0 ]] && return 1

  case "$choice" in
    ADD) 	Append 	;;
    EDIT) 	Edit    ;;
    REFRESH) 	Refresh ;;
    REMOVE) 	Remove 	;;
    RESTORE)    Restore "$changed" ;;
    APPLY) 	Apply 	;;
  esac
  return 0
}

# -------------------------------------------------------------------------------

marquees() {
  declare -A paths=()
  declare -a missing=()
  declare -a items=()
  declare -a titles=()
  declare -a files=()

  local WIDTH=$(cat /sys/class/graphics/fb0/virtual_size | cut -d, -f1)
  local IMG_ZIP="$HOMEDIR/ADMIN/FILES/marquees.zip"
  local name=''; local title=''; local file=''; local mode=''; local item_list=''
  local preview_time=5
  local img_h=0
  local found=0
  local count=0
  local code=0
  local rc=0

  clear

  [ ! -f "$temp_dir/.list" ] && cp -f "$assets_path/GAMES/.list" "$temp_dir/.list"; total=$(cat "$temp_dir/.list" | wc -l)
  [ ! -f "$DIR/assets/marquees.list" ] && 7z l "$IMG_ZIP" | grep '^.*.jpg' | awk '{$1=$1;print $NF}' | sort > "$DIR/assets/marquees.list"
  printf '' > "$temp_dir/marquees_missing"

  choice=$(sudo -u pi bash -c 'dialog --colors --title "FILTER MODE" --cancel-label "Cancel" --ok-label "MAN" --extra-button --extra-label "AUTO" --menu "\n Select page filter and choose mode:\n\n MAN  = MANUALLY ADD ITEMS\n AUTO = SELECT ALL IN LIST\n -" 12 42 0 "ALL" "" "LISTED" "" "FAV" ""' 2>&1 1>$(tty)); code=$?; [[ ! $code == @(0|3) ]] && code=1; clear
  [ $code -eq 1 ] && return 0 || mode="$([ $code -eq 3 ] && echo AUTO || echo MAN)"

  case "$choice" in
    'FAV') 	item_list=$(awk -F'\x27' '$14==0 && $16 ~ /FAV/ {print $2}' "$temp_dir/.list") ;;
    'LISTED') 	item_list=$(awk -F'\x27' '$14==0{gsub(/[][ ]/,"",$16); if (length($16)) print $2;}' "$temp_dir/.list") ;;
    *) 		item_list=$(awk -F'\x27' '$14==0{print $2}' "$temp_dir/.list") ;;
  esac

  if   [[ $mode == MAN ]]; then
    for i in $(echo "$item_list" | sort); do items+=("$i" "" OFF); done
    missing=($(whiptail --checklist "$(echo -e "\nFILTER: $choice")" --title "MISSING MARQUEES" 0 0 0 "${items[@]}" 2>&1 1>$(tty))); [ $? -ne 0 ] && return 0
    missing=($(echo "${missing[@]}" | sed 's/\x22//g;s/[ ]/\n/g'));
  elif [[ $mode == AUTO ]]; then # AUTO
    missing=($(echo "$item_list" | sort))
    whiptail --yesno "AUTO FIND $choice MARQUEES?" 0 0; [ $? -ne 0 ] && return 0
  fi

  for (( i=0; i<${#missing[@]}; i++ )); do name="${missing[$i]}"
    file="$(grep "^$name" $DIR/assets/marquees.list)"
    [[ ${#file} -eq 0 ]] && file="$(grep "^${name%?}" $DIR/assets/marquees.list)"
    [[ ${#file} -ge 1 ]] && { paths["$name"]="$file"; files=($(echo "${paths[$name]}")); titles+=("$name" "$(printf '%-14s (%d)\n' ${files[0]} ${#files[@]})"); } || { echo "$name" >> "$temp_dir/marquees_missing"; }

  done

  if [[ ${#missing[@]} -gt 0 ]]; then code=0
    local imgs_missing=$(cat "$temp_dir/marquees_missing" | wc -l)

    for dir in img ppm; do [ ! -d "$temp_dir/marquees/$dir" ] && mkdir -p -m 775 "$temp_dir/marquees/$dir"; done

    while [[ $code -eq 0 ]]; do items=()
      name=$(whiptail --menu "$(echo -e "\nFILTER:  $choice\nMISSING: $(cat $temp_dir/marquees_missing | wc -l)")" --title "VERIFY FILES" --ok-button "Edit" --cancel-button "Done" 0 0 0 "${titles[@]}" 2>&1 1>$(tty)); code=$?

      if [ $code -eq 0 ]; then files=($(echo "${paths[$name]}")); items=(); rc=3
        while [[ $rc -eq 3 ]]; do
          i=$(sudo -u pi str="${files[*]}" bash -c 'items=(); files=($(echo "$str")); for ((i=0; i<${#files[@]}; i++)); do [ $i -eq 0 ] && items+=("$i" "[*] ${files[$i]}") || items+=("$i" "[ ] ${files[$i]}"); done; dialog --colors --extra-button --extra-label "Preview" --ok-label "Select" --menu "MENU" 0 0 0 "${items[@]}"' 2>&1 1>$(tty)); rc=$?; clear
          if [[ $rc -eq 3 ]]; then
            [ ! -d "$temp_dir/marquees/previews" ] && mkdir -p -m 775 "$temp_dir/marquees/previews"; file="$temp_dir/marquees/previews/${files[$i]}"
            [ ! -f "$file" ] && unzip -o -d "$temp_dir/marquees/previews" "$IMG_ZIP" "${files[$i]}" &> /dev/null
            sudo fbi -T 2 -d /dev/fb0 -a -t $preview_time --once --noverbose "$file"; sleep $preview_time
          fi
        done

        if [ $rc -eq 0 ] && [ $i -gt 0 ]; then titles=()
          n="${files[$i]}"; files["$i"]="${files[0]}"; files[0]="$n"; paths["$name"]="$(echo "${files[@]}")"
          for (( i=0; i<${#missing[@]}; i++ )); do name="${missing[$i]}"; files=($(echo "${paths[$name]}")); [[ ${#files[@]} -gt 0 ]] && titles+=("$name" "'${files[0]}' (${#files[@]})"); done
        fi
      fi
    done

    if [[ ${#paths[@]} -gt 0 ]]; then
      whiptail --yesno "BEGIN CONVERSION?" 0 0; [ $? -ne 0 ] && return 0

      for key in "${!paths[@]}"; do files=($(echo "${paths[$key]}")); file="${files[0]}"; unzip -o -d "$temp_dir/marquees/img" "$IMG_ZIP" "$file" &> /dev/null; done
      for tmp in $temp_dir; do sudo chmod -R 775 $temp_dir; sudo chown -R pi:root $temp_dir; done

      local file_count=$(find "$temp_dir/marquees/img" -name '*.jpg' | wc -l)
      paths=(); rc=3; prog 0 $file_count

      while read file; do (( count++ )); prog $count "Converting.."
        types=($(file -b --extension "$file" | sed 's/\// /g'))

        #[[ "${types[@]}" =~ png ]] && gm convert -resize ${WIDTH}x +profile "*" "${file}" "${file}.converted" &>/dev/null || gm convert -resize ${WIDTH}x "${file}" "${file}.converted" &>/dev/null
        gm convert -resize ${WIDTH}x "${file}" "${file}.converted" &>/dev/null

        if [ -f "${file}.converted" ]; then
          img_h=$(identify -format "%h" "${file}.converted")
          name="$([[ $file =~ .*/(.*)[.].* ]] && echo ${BASH_REMATCH[1]})"
          paths["$name"]="$temp_dir/marquees/img/${name}_${img_h}.jpg"
          mv -f "${file}.converted" "${paths[$name]}"; rm -f "$file"
        fi
      done< <(find "$temp_dir/marquees/img" -maxdepth 1 -name '*.jpg')

      while [[ $rc -eq 3 ]]; do
        file=$(sudo -u pi dir="$temp_dir/marquees/img" bash -c 'items=(); files=($(find $dir -maxdepth 1 -name *.jpg | grep -o "[^/]*$" | sort)); for f in "${files[@]}"; do items+=("$f" ""); done; dialog --colors --extra-button --extra-label "Edit" --menu "SCALED RESULTS" 0 32 0 "${items[@]}"' 2>&1 1>$(tty)); rc=$?
        if [[ $rc -eq 3 ]]; then
          marquee_editor "$temp_dir/marquees/img/$file"
          if [[ $(cat "$temp_dir/marquees/edits.log" | wc -l) -gt 0 ]]; then
            file=$(cat "$temp_dir/marquees/edits.log")
            name="$([[ $file =~ .*/(.*)[_].* ]] && echo ${BASH_REMATCH[1]})"
            paths["$name"]="$file"
          fi
        fi
      done

      whiptail --yesno "$(echo -e "\n${#paths[@]} file(s) will be converted:\n\n$(for p in ${paths[@]}; do echo "- ${p##*/}"; done)")" --title "CONFIRM" --yes-button "START" --no-button "ABORT" 0 0 2>&1 1>$(tty)

      if [ $? -eq 0 ]; then clear
        for path in "${paths[@]}"; do
          name="$([[ $path =~ .*/(.*)[.].* ]] && echo ${BASH_REMATCH[1]})"
          file="$temp_dir/marquees/ppm/$name.ppm"
          [ -f "$path" ] && { fbConvert "$path" "$file" &>/dev/null; path["$name"]="$file"; }
        done
      fi

      for tmp in $tmp; do sudo chmod -R 775 $tmp; sudo chown -R pi:root $tmp; done
    fi
  fi

  return 0
}

# ----------------------------------------------------------------------------------------------

marquee_editor() {
  [ $# -eq 0 ] && return 1

  local file="${1}"; [ ! -f "$file" ] && return 1
  local file_orig="$file"
  local name="$([[ $file =~ .*/(.*)[_].* ]] && echo ${BASH_REMATCH[1]})"
  local img_path="$HOMEDIR/ADMIN/FILES/IMAGES/MARQUEES"
  local img_w=$(cat /sys/class/graphics/fb0/virtual_size | cut -d, -f1)
  local output_cropped=''
  local new_val=''
  local source=''
  local output=''
  local changed=0
  local status=0
  local img_h=0
  local y_top=0
  local y_bot=0
  local code=0

  for dir in "$temp_dir/marquees/img/edited"; do [ ! -d $dir ] && mkdir -p -m 775 $dir; sudo chmod -R 775 $dir; sudo chown -R pi:root $dir; done
  output="$temp_dir/marquees/img/edited/${file##*/}"; cp -f "$file" "$output"; img_h=$(identify -format "%h" "$output"); y_top=$img_h

  printf '' > "$temp_dir/marquees/edits.log"

  while [[ $code -eq 0 ]]; do
    action=$(whiptail --ok-button "Select" --cancel-button "Done" --menu "edit image for '$name'" 0 0 0 "PREVIEW" "" "CROP IMAGE" "" "NEW SOURCE" "" 2>&1 1>$(tty)); code=$?

    if [ $code -eq 0 ]; then
      case "$action" in
        'PREVIEW')    sudo fbi -T 2 -d /dev/fb0 -t 0 --once --noverbose "${output_cropped:-$output}" &>/dev/null ;;
        'CROP IMAGE') status=0
          while [[ $status -eq 0 ]]; do
            crop_mode=$(whiptail --title "CROP IMAGE" --menu "edit ${output##*/} ($img_wx$img_h)" 0 0 0 "TOP ($y_top)" "" "BOTTOM ($y_bot)" "" "APPLY" "" 2>&1 1>$(tty)); status=$?
            if [ $status -eq 0 ]; then changed=1
              [[ ${#output_cropped} -gt 0 ]] && { output="$temp_dir/marquees/img/edited/${file##*/}"; cp -f "$file" "$output"; output_cropped=''; }
              case "${crop_mode%% *}" in
                'TOP')    new_val=$(sudo -u pi img_h="$img_h" y_top="$y_top" y_bot="$y_bot" bash -c 'dialog --colors --title "CROP IMAGE" --rangebox "TOP" 0 0 $y_bot $img_h $y_top' 2>&1 1>$(tty)); [[ $? -eq 0 ]] && y_top=${new_val//[[:blank:]]/} ;;
                'BOTTOM') new_val=$(sudo -u pi img_h="$img_h" y_top=$y_top y_bot=$y_bot bash -c 'dialog --colors --title "CROP IMAGE" --rangebox "BOTTOM" 0 0 0 $y_top $y_bot' 2>&1 1>$(tty)); [[ $? -eq 0 ]] && y_bot=${new_val//[[:blank:]]/} ;;
                'APPLY')  whiptail --title "APPLY CROP?" --yesno "TOP : $((img_h-y_top))px\nBOT.: ${y_bot}px" 0 0; [ $? -eq 0 ] && { status=1; output_cropped="${output/.jpg/}-cropped.jpg"; gm convert "$output" -crop "$img_w"x"$((img_h-y_bot))"+0+"$((img_h-y_top))" "$output_cropped" &>/dev/null; } ;;
              esac
            fi
          done
	;;
        'NEW SOURCE')
          declare -a sources=(); declare -a types=()

          for img in $(find "$img_path" -print | file -if - | grep "image" | cut -d: -f1 | sort -t \/); do sources+=("${img##*/}" ""); done
          source=$(whiptail --menu "MENU" 0 0 0 "${sources[@]}" 2>&1 1>$(tty))

          if [ $? -eq 0 ]; then changed=1
            for dir in "$temp_dir/marquees/img/sourced"; do [ ! -d $dir ] && mkdir -p -m 775 $dir; sudo chmod -R 775 $dir; sudo chown -R pi:root $dir; done
            source="$temp_dir/marquees/img/sourced/$source"; cp -f "$img_path/${source##*/}" "$source"; types=($(file -b --extension $source | sed 's/\// /g'))

            [[ "${types[@]}" =~ png ]] && gm convert -resize ${img_w}x +profile "*" "${source}" "${source/.jpg/}-scaled.jpg" &>/dev/null || gm convert -resize ${img_w}x "${source}" "${source/.jpg/}-scaled.jpg" &>/dev/null

            img_h=$(identify -format "%h" "${source/.jpg/}-scaled.jpg")
            file="$temp_dir/marquees/img/edited/${name}_${img_h}-sourced.jpg"
            mv -f "${source/.jpg/}-scaled.jpg" "$file"; cp -f "$file" "$output"; mv -f "$output" "${file/-sourced/}"

            output="${file/-sourced/}"; output_cropped=''; y_top=$img_h; y_bot=0
          fi
	;;
      esac
    fi
  done

  if [[ $changed -eq 1 ]]; then
    whiptail --yes-button "Keep" --no-button "Discard" --yesno "Keep Changes?" 0 32

    if [ $? -eq 0 ]; then
      file="$temp_dir/marquees/img/${name}_$(( img_h-((img_h-y_top)+y_bot) )).jpg"
      cp -f "${output_cropped:-$output}" "$file"; rm -f "$file_orig"; echo "${file}" > "$temp_dir/marquees/edits.log"
    fi
  fi

  rm -Rf "$temp_dir/marquees/img/edited"

  return 0
}

# ----------------------------------------------------------------------------------------------

Edit() {
  declare -A game_data=()
  declare -a entries=()
  declare -a changed=()
  local arg="$@"
  local game=''
  local input_str=''
  local index=0
  local edit_done=0
  local count=0
  local typer_file="$joytyper_dir/IO/output"

  [ ! -f "$temp_dir/.list" ] && cp -f "$assets_path/GAMES/.list" "$temp_dir/.list"; total=$(cat "$temp_dir/.list" | wc -l);
  for file in table timers clearram; do [ ! -f "$temp_dir/cafca_$file" ] && cp -f "$cafca_dir/$file" "$temp_dir/cafca_$file"; done

  if [[ "$arg" =~ ^[0-9]+$ ]]; then
    data=$(awk -v i="$arg" 'NR==i' "$temp_dir/.list"); data=$(format_table "$data")
  else
    clear; whiptail --yesno "" --title "EDIT MODE" --yes-button "SIMPLE" --no-button "DETAILED" 5 56 2>&1 1>$(tty)

    if [ $? -eq 0 ]; then
      while read line; do entries+=("${line%% *}" "${line#* }"); done< <( awk -F'\x27' -v w="$(tput cols)" '{gsub(/ /,"",$1); print $1" "substr($4,1,(w/2)),"("$2")"}' "$temp_dir/.list")
      index=$(whiptail --menu "" --title "GAMES" --ok-button "Select" 0 0 0 "${entries[@]}" 2>&1 1>$(tty)); [ $? -ne 0 ] && return $?
      data=$(awk -v i="$index" 'NR==i' "$temp_dir/.list"); data=$(format_table "$data")
    else prog 0 $total
      while read line; do entries+=("$line" ""); (( count++ )); prog $count; done< <( format_table "$temp_dir/.list" )
      data=$(whiptail --menu "$(echo '~# NAME TITLE YEAR TYPE PAGES D M C T' | awk '{printf("%-5s %-9s %-8s %-4s %-2s %-6s %-1s %-1s %-1s %-2s\n",$1,$2,$3,$4,$5,$6,$7,$8,$9,$10)}')" --ok-button "Select" 0 0 0 "${entries[@]}" 2>&1 1>$(tty)); [ $? -gt 0 ] && return $?
    fi
  fi

  declare -a labels=('' 'index' 'name' '' 'title' '' 'year' '' 'publish' '' 'type' '' 'path' '' 'marquee' '' 'pages' 'timer' 'clear' 'cafca' 'config')

  while read line; do game_data["${labels[${line%% *}]^^}"]="$(echo "$line" | cut -d ' ' -f2-)"; done< <(awk -F '\x27' -v l="${data%% *}" 'NR == l { for (i=2;i<=NF;i+=2) print i" "$i; }' "$temp_dir/.list")
  while read line; do game_data["${line%% *}"]="${line##* }"; done< <(echo "$data" | awk '{print "TIMER "$NF; print "CLEAR "$(NF-1); print "CAFCA "$(NF-3)}')

  while [[ $edit_done -eq 0 ]]; do
    entries=()
    for (( i=2; i<${#labels[@]}; i++ )); do k="${labels[$i]^^}"
      if [[ ${#k} -gt 0 ]]; then val="${game_data[$k]}"
        [[ $k == CAFCA ]] && [[ ${val:0:1} == $'\x78' ]] && val="$(grep -w -m1 ${game_data[NAME]} $temp_dir/cafca_table | awk '{print $2}')"
        [[ $k == PATH ]]  && val="${val//$roms_dir/}"
        [[ $k == TIMER ]] && val="${val/-1/w}"
        [[ $k == PAGES ]] && val=$(echo "$val" | sed 's/\[ *[^[:alnum:]]*//g;s/[^[:alnum:]]*\( *\|\]\)$//;s/ /,/g')
        [[ $k == CONFIG ]] && val=$(find $roms_dir -type f -iname "${game_data[NAME]}.zip.cfg" -exec false {} +; echo "$?")
        game_data[$k]="${val:-${game_data[$k]}}"
        entry_key="${k} $([[ ${changed[@]} =~ $k ]] && echo '*' || echo ' ')"
        entries+=("$entry_key " "${game_data[$k]}");
      fi
    done

    choice=$(whiptail --menu "MENU" --ok-button "Edit" --cancel-button "Return" 20 0 0 "${entries[@]}" 2>&1 1>$(tty)); edit_done=$?;

    [[ $edit_done -gt 0 ]] && break

    choice="${choice%% *}"

    case "$choice" in
      'NAME'|'TITLE'|'PUBLISH'|'YEAR'|'PATH'|'CAFCA'|'CONFIG')
          declare -a lines_changed=()
          local typer_output=''
          local typer_data="${game_data[$choice]}"
          local file_edited=0

          [[ $choice == CAFCA ]] && typer_data="$(grep -w -m1 ${game_data[NAME]} $temp_dir/cafca_table | awk '{$1=$1;print}')"

          if [[ $choice == CONFIG ]]; then
            [ ! -d "$temp_dir/cfg" ] && mkdir -p -m 775 "$temp_dir/cfg"
            [ ! -f "$temp_dir/cfg/${game_data[NAME]}.zip.cfg" ] && { for cfg in "${game_data[NAME]}.zip.cfg"; do touch "$temp_dir/cfg/$cfg"; cp -f "$roms_dir/$cfg" "$temp_dir/cfg/$cfg"; typer_data="$temp_dir/cfg/$cfg"; done; }; cfg="$temp_dir/cfg/${game_data[NAME]}.zip.cfg"

            while [[ $file_edited -eq 0 ]]; do entries=()
              while read line; do key=${line%% *}; [[ ${lines_changed[@]} =~ $key ]] && key="$key*"; entries+=("$key" "${line:2}"); done< <(awk '{print NR" "$0}' "$cfg"); entries+=("" "< ADD LINE >")
              line_choice=$(whiptail --menu "${cfg##*\/}" 0 0 0 "${entries[@]}" 2>&1 1>$(tty)); file_edited=$?; index=${line_choice//\*/}

              if [[ $file_edited -eq 0 ]]; then changed+=("$choice");
                if [[ -z $index ]]; then
                  printf '\n' | sudo tee -a "$cfg" &>/dev/null; lines_changed+=("$(cat $cfg | wc -l)")
                else
                  typer_data=$(sed -n "${index}p" "$cfg")
                  bash "$joytyper_dir/joytyper.sh" "$typer_data" "EDIT ${cfg##*\/}:"
                  typer_output="$(cat $typer_file)"
                  [[ ${#typer_output} -gt 0 ]] && [[ ! "$typer_output" == "$typer_data" ]] && { sed -i "${index} s/^.*$/${typer_output}/" "$cfg"; lines_changed+=("$index"); }
                fi
              else
                break
              fi
            done
          else
            bash "$joytyper_dir/joytyper.sh" "$typer_data" "EDIT $choice:"; typer_output="$(cat $typer_file)"
            [[ ${#typer_output} -gt 0 ]] && [[ ! "$typer_output" == "${game_data[$choice]}" ]] && { game_data["$choice"]="$typer_output"; changed+=("$choice"); }
          fi
          ;;&
      'TYPE')
          entries=()
          for type in JOY GUN SPIN; do [[ $type == ${game_data[TYPE]} ]] && entries+=("$type" "" ON) || entries+=("$type" "" OFF); done
          result=$(whiptail --radiolist "SELECT $choice" 0 0 0 "${entries[@]}" 2>&1 1>$(tty))
          [ $? -eq 0 ] && { game_data["$choice"]="$result"; changed+=("$choice"); }
	  ;;
      'PAGES')
          entries=()
          declare -a all_pages=($(IFS=$'\n'; sed -n '/\[ \([^]]*\) \]/ { s/.*\[ \([^]]*\) \].*/\1/; s/ /\n/g;/^$/d; p }' "$temp_dir/.list" | sort -u))
          for page in "${all_pages[@]}"; do [[ $(IFS=,; echo "${game_data[PAGES]}") =~ $page ]] && entries+=("$page" "" ON) || entries+=("$page" "" OFF); done; entries+=("<ADD>" "" OFF)
          result=$(whiptail --checklist "SELECT $choice" 0 0 0 "${entries[@]}" 2>&1 1>$(tty))

          if [ $? -eq 0 ]; then result=$(echo "$result" | sed 's/\x22//g;')
            if [[ "$result" =~ "<ADD>" ]]; then result=$(echo "${result/<ADD>/}" | sed 's/ $//'); typer_output=''
              bash "$joytyper_dir/joytyper.sh" "$typer_data" "TYPE PAGE NAME:"; [ $? -eq 0 ] && typer_output="$(cat $typer_file)"

              if [[ ${#typer_output} -gt 0 ]]; then local new_page="$typer_output"
                result=$(echo "${new_page} ${result}" | sed 's/ $//'); new_pages["$new_page"]="$new_page"; whiptail --yesno "UPDATE PAGE TITLE?" 0 0 2>&1 1>$(tty)
                [ $? -eq 0 ] && { bash "$joytyper_dir/joytyper.sh" "$new_page" "TYPE PAGE TITLE:"; [ $? -eq 0 ] && new_pages["$new_page"]="$(cat $typer_file)"; }
              fi
            fi
            [[ ${#result} -gt 0 ]] && { result="${result/ /,}"; game_data["$choice"]="$result"; changed+=("$choice"); }
          fi
          ;;
      'TIMER')
          local new_val="${game_data[$choice]/w/-1}"
          local timer_choice=$(sudo -u pi choice="$choice" val="$new_val" bash -c 'dialog --colors --title "SET $choice" --cancel-label "Cancel" --ok-label "Change" --extra-button --extra-label "Write-mode" --yesno "$(echo "\nVALUE: $val\n\n* >1 = timer on\n*  1 = disabled\n*  0 = timer off\n* -1 = writeable")" 16 0' 2>&1 1>$(tty); echo $?); clear
	  case $timer_choice in
              0) new_val=$(sudo -u pi choice="$choice" val="$new_val" bash -c 'dialog --colors --rangebox "$choice" 0 0 0 60 $val' 2>&1 1>$(tty); [ $? -ne 0 ] && echo "$new_val" ); changed+=("$choice"); clear ;;
              3) new_val='-1'; changed+=("$choice") ;;
          esac
          game_data["$choice"]="${new_val/-1/w}"
          ;;
      'MARQUEE'|'CLEAR')
          if [[ $choice == MARQUEE ]]; then
            if [[ ${game_data[$choice]} -eq 1 ]]; then
              find "$assets_path/media/marquees/" -name '${game_data[NAME]}_*.ppm' -exec false {} +;
              [ $? -eq 1 ] && whiptail --yesno "DELETE MARQUEE FILE?" --title "WARNING" 0 0 2>&1 1>$(tty)
              [ $? -eq 0 ] && find "$assets_path/media/marquees/" -name '${game_data[NAME]}_*.ppm' -exec rm -f {} \;
            else
              whiptail --yesno "Set from pre-existing?" 0 0 #2>&1 1>$(tty)
              if [ $? -eq 0 ]; then entries=()
                for f in $(ls -1 "$assets_path/media/marquees" | grep '.*[.]ppm'); do entries+=("$f" ""); done
                local new_val=$(whiptail --menu "MENU" 0 0 0 "${entries[@]}" 2>&1 1>$(tty))
                [ $? -eq 0 ] && { \
                  for dir in img ppm; do [ ! -d "$temp_dir/marquees/$dir" ] && mkdir -p -m 775 "$temp_dir/marquees/$dir"; done; \
                  cp -f "$assets_path/media/marquees/$new_val" "$temp_dir/marquees/ppm/${game_data[NAME]}_${new_val/*[_]/}"; \
                }
              fi
            fi
          fi
          game_data["$choice"]=$(( ! ${game_data[$choice]} )); changed+=("$choice")
	  ;;
    esac
  done

  if [[ ${#changed[@]} -gt 0 ]]; then
    whiptail --yesno "SAVE CHANGES?" 0 0 2>&1 1>$(tty); [ $? -ne 0 ] && return 0

    for key in "${!game_data[@]}"; do
      if [[ "${changed[@]}" =~ "$key" ]]; then
        data="${game_data[$key]}"; name="${game_data[NAME]}"
        case "$key" in
            'CAFCA'|'TIMER'|'CLEAR')
              [[ $key == TIMER ]] && data=${data/w/-1}
              [[ ! $key == CAFCA ]] && data="$name $data"
              file=$(echo "$key" | awk '{print (($1=="CLEAR")?"clearram":($1=="CAFCA")?"table":($1=="TIMER")?"timers":"")}'); file="$temp_dir/cafca_$file"
              sed -i "/^${name}\( \|$\)/d" $file; echo "$data" | sudo tee -a $file; sort -k 1,1 $file | column -t | sudo tee $file &>/dev/null
              ;;
            *)
              local list_col=; local index=$(grep -w -m1 "$roms_dir${game_data[PATH]}" "$temp_dir/.list" | awk '{print $1}')
              [[ $key == PAGES ]] && data="[ ${data//,/ } ]"
              for (( i=2; i<${#labels[@]}; i++ )); do [[ "${labels[$i]^^}" == "$key" ]] && { list_col="$((i/2))"; break; }; done; #read -srn1 void >/dev/null
              sed -i "${index}{s/'[^']*'/'${data}'/${list_col}}" "$temp_dir/.list"
	      ;;
        esac
      fi
    done
    [[ "$arg" =~ ^[0-9]+$ ]] && return 1
  fi
  return 0
}

# -------------------------------------------------------------------------------

Append() {
  local choice=''

  [ $# -gt 0 ] && [[ "${1^^}" == @(DOWNLOAD|SCRAPE|MARQUEES) ]] && choice="${1^^}"
  [ ! -f "$temp_dir/downloads" ] && printf '' > "$temp_dir/downloads"

  dls="$(cat $temp_dir/downloads | wc -l)"
  clear

  if [[ ${#choice} -eq 0 ]]; then
    choice=$( \
      whiptail --menu "" --title "ADD DATA" 0 0 0 \
        "DOWNLOAD" "" \
        "SCRAPE" "$([[ $dls -gt 0 ]] && echo ' *')" \
        "MARQUEES" "" 2>&1 1>$(tty))

    [ $? -gt 0 ] && return 1
  fi

  case "$choice" in
    DOWNLOAD)   [[ "${global_funcs[@]}" =~ downloader ]] &&  downloader ;;
    SCRAPE)     [[ "${global_funcs[@]}" =~ scraper ]]    &&  scraper;    [ $? -eq 0 ] && return 0 ;;
    MARQUEES)   [[ "${global_funcs[@]}" =~ marquees ]]   &&  marquees;   [ $? -eq 0 ] && return 0 ;;
  esac

  Append
}

# -------------------------------------------------------------------------------

Refresh() {
  declare -a ids=()

  clear; echo "REFRESHING GAME LIST.." | log_append
  [ ! -f "$temp_dir/.list" ] && cp -f "$assets_path/GAMES/.list" "$temp_dir/.list"

  # DECODE HTML CHARACHTERS
  sed -i 's/&apos;//' "$temp_dir/.list"
  perl -i -MHTML::Entities -pe 'decode_entities($_);' "$temp_dir/.list"

  # CHECK ORPHAN ENTRIES
  #echo -e "\nORPHAN ENTRIES:\n" | log_append_silent
  for game in $(awk -F'\x27' '{print $2}' "$temp_dir/.list" | sort); do
    find "$roms_dir/" -maxdepth 1 -name "$game.zip" -exec false {} +;
    [ $? -eq 0 ] && ids+=($(grep -m1 "$game.zip" "$temp_dir/.list" | awk '{print $1}'))
  done
  if [[ ${#ids[@]} -gt 0 ]]; then echo "${ids[@]}" | log_append_silent
    for i in "${ids[@]}"; do sed -i "/^${i}/d" "$temp_dir/.list"; done
    awk '{$1="";print NR""$0}' "$temp_dir/.list" > "$temp_dir/.list.new" && cp -f "$temp_dir/.list.new" "$temp_dir/.list"
    #awk '{$1=""; print NR$0}' "$temp_dir/.list" | sudo tee "$temp_dir/.list" &>/dev/null
  fi

  # CHECK DUPLICATE LINES
  #echo -e "\nDUPLICATE LINES:\n" | log_append_silent
  while read row; do
    sed -i "/${row%% *}/d" "$temp_dir/.list"; echo "$row" | log_append_silent
  done< <(awk -F'\x27' 'seen[$12]++' "$temp_dir/.list")
  awk '{$1="";print NR""$0}' "$temp_dir/.list" > "$temp_dir/.list.new" && cp -f "$temp_dir/.list.new" "$temp_dir/.list"
  #awk '{$1=""; print NR$0}' "$temp_dir/.list" | sudo tee "$temp_dir/.list" &>/dev/null

  # REARRANGE LEADING ´THE´ IN TITLE
  while read row; do
    local i="${row%% *}"; local title="${row#* }"
    sed -i "${i}{s/'[^']*'/'${title}'/2}" "$temp_dir/.list"
  done< <(awk -F'\x27' '$4 ~ /^The/ {gsub(/^The /,"",$4); gsub(/ \(.*\)$/,"",$4); print $1$4", The"}' "$temp_dir/.list")
  sort -k4,4 -t \' "$temp_dir/.list" --output "$temp_dir/.list"
  awk '{$1="";print NR""$0}' "$temp_dir/.list" > "$temp_dir/.list.new" && cp -f "$temp_dir/.list.new" "$temp_dir/.list"
  #awk '{$1="";print NR""$0}' "$temp_dir/.list" | sudo tee "$temp_dir/.list" &>/dev/null

  # CHECK DUPLICATE TITLES
  while read row; do
    local title=$(awk -F'\x27' -v n="$row" 'NR==n { print $4" ("$2")" }' "$temp_dir/.list")
    sed -i "${row}{s/'[^']*'/'${title}'/2}" "$temp_dir/.list"
  done< <(awk -F'\x27' 'seen[$4]++' "$temp_dir/.list" | awk '{print $1}')

  # CHECK FOR MARQUEE AND SET FLAG
  for i in $(seq 1 $(cat "$temp_dir/.list" | wc -l)); do
    name=$(sed -n "${i}p" "$temp_dir/.list" | cut -d\' -f2)
    find "$assets_path/media/marquees" -type f -iname "$name[_]*.ppm" -exec false {} +; found=$?;
    sed -i "${i}{s/'[^']*'/'${found}'/7}" "$temp_dir/.list"
  done

  update_lists
  Apply
}

# -------------------------------------------------------------------------------

Remove() {
  declare -a entries=()

  [ ! -f "$temp_dir/.list" ] && cp -f "$assets_path/GAMES/.list" "$temp_dir/.list"

  choice=$(whiptail --title "REMOVE" --menu "" 0 0 0 "ROMS" "" "MARQUEES" "" 2>&1 1>$(tty)); [ $? -ne 0 ] && return 1

  case "$choice" in
    'MARQUEES')
        local dir="$assets_path/media/marquees"
        while read f; do entries+=("${f%%_*}" "  ${f##*_}" OFF); done< <(find $dir -maxdepth 1 -name '*.ppm' | grep -o '[^/]*$' | cut -d. -f1 | sort)
        declare -a imgs=($(whiptail --title "REMOVE" --checklist "Marquees:" 0 0 0 "${entries[@]}" 2>&1 1>$(tty)))

        if [ $? -eq 0 ] && [[ ${#imgs[@]} -gt 0 ]]; then
          imgs=($(echo "${imgs[@]}" | sed 's/\x22//g;s/[ ]/\n/g'))
          whiptail --title "WARNING" --defaultno --yesno "Delete ${#imgs[@]} marquees?\n\n$(echo "${imgs[@]}\n" | sed 's/ /\n/g')" 0 0
          if [ $? -eq 0 ]; then
            for img in "${imgs[@]}"; do
              find $dir -maxdepth 1 -type f -name "${img}_*.ppm" -exec rm -f {} \;
            done
          fi
        fi
    ;;
    'ROMS')
      while read line; do
        name="${line%%.*}"; entries+=("$name" "$([[ $(grep -c $name $temp_dir/.list) -eq 0 ]] && echo '*')")
      done< <(find $roms_dir -type f -iname '*.zip' -print | grep -o '[^/]*$' | sort)

      game=$(whiptail --menu "$(echo -e "\nSELECT GAME"):" --title "REMOVE" 0 0 0 "${entries[@]}" 2>&1 1>$(tty))

      if [ $? -eq 0 ]; then
        whiptail  --yesno "$(echo -e "\n  REMOVE DATA FOR '$game' ?")" --title "! WARNING !" 0 0 2>&1 1>$(tty)
        if [ $? -eq 0 ]; then
          for file in .list .shooters .spinners; do [ ! -f "$temp_dir/$file" ] && cp -f "$assets_path/GAMES/$file" "$temp_dir/$file"; sed -E -i "/${game}[' ]*/d" "$temp_dir/$file"; done
          for file in .list .shooters .spinners; do [ ! -f "$temp_dir/$file" ] && cp -f "$assets_path/GAMES/$file" "$temp_dir/$file"; sed -E -i "/${game}[' ]*/d" "$temp_dir/$file"; done
          for file in table timers clearram; do [ ! -f "$temp_dir/cafca_$file" ] && cp -f "$cafca_dir/$file" "$temp_dir/cafca_$file"; sed -E -i "/${game}[' ]*/d" "$temp_dir/cafca_$file"; done
          awk '{$1="";print NR""$0}' "$temp_dir/.list" | sudo tee "$temp_dir/.list" &>/dev/null

          [[ $(whiptail  --yesno "$(echo -e "\n  DELETE ROM '$game' ?")" --title "! WARNING !" 0 0 2>&1 1>$(tty); echo $?) -eq 0 ]] && sudo rm -fv "$roms_dir/$game.zip" | log_append
        fi
      fi
    ;;
  esac
  return 0
}

# -------------------------------------------------------------------------------

Restore() {
  local changed="${1}"
  declare -a paths=()
  declare -a titles=()
  declare -a items=()

  [[ $(find "$DIR/.backup/" -name '*.zip' | wc -l) -lt 1 ]] && { whiptail --msgbox "    NO ARCHIVES FOUND!" 0 0 2>&1 1>$(tty); return 1; }

  while read line; do paths+=("${line##*/ }"); titles+=("${line%% /*}"); done< <(ls -lt "$DIR/.backup" | awk '/backup/ {raw=$NF; gsub(/backup.|.zip/,"",$NF); print substr($NF,1,4)"-"substr($NF,5,2)"-"substr($NF,7,2)" "substr($NF,10,2)":"substr($NF,12,2)" / "raw}')
  for (( i=0; i<${#titles[@]}; i++ )); do items+=("$i" "${titles[$i]}"); done

  choice=$(whiptail --menu "" --title "RESTORE" 0 0 0 "${items[@]}" 2>&1 1>$(tty)); [ $? -ne 0 ] && return 1
  whiptail --yesno "Extract files in '${paths[$choice]}' ?" 0 0 2>&1 1>$(tty)

  if [ $? -eq 0 ]; then
    [[ $changed -eq 1 ]] && whiptail --yesno "Discard changes?" 0 0 2>&1 1>$(tty); [ $? -ne 0 ] && return 1
    mkdir -p -m 775 "$temp_dir/restored"; 7z x -o"$temp_dir/restored" "$DIR/.backup/${paths[$choice]}" | log_append_silent
    for tmp in $temp_dir; do sudo chmod -R 775 $temp_dir; sudo chown -R pi:root $temp_dir; done
    for file in table timers clearram; do find "$temp_dir/restored/" -name "$file" -exec mv -f {} "$temp_dir/cafca_$file" \; ;done
    for file in .list .shooters .spinners; do find "$temp_dir/restored/" -name "$file" -exec mv -f {} "$temp_dir/" \; ;done
  fi
}

# -------------------------------------------------------------------------------

Apply() {
  local arc_name="backup.$(date '+%Y%m%d_%H%M%S').zip"

  whiptail --yesno "APPLY CHANGES?" 0 0 2>&1 1>$(tty)

  if [ $? -eq 0 ]; then clear;
    [ ! -d "$DIR/.backup" ] && mkdir -p -m 775 "$DIR/.backup"

    [ -d "$temp_dir/cfg" ] && find "$temp_dir/cfg/" -name '*.zip.cfg' -exec bash -c '7z a "${2}" "${1}/${0##*/}"; cp -f "${0}" "${1}/"' {} "$roms_dir" "$DIR/.backup/$arc_name" \; | log_append_silent
    #[ -d "$temp_dir/marquees/ppm" ] && find "$temp_dir/marquees/ppm" -name '*.ppm' -exec cp -fv {} "$assets_path/media/marquees/" \; | log_append_silent

    if [ -d "$temp_dir/marquees/ppm" ]; then
      for m_path in $(find "$temp_dir/marquees/ppm" -maxdepth 1 -name '*.ppm' | sort); do
        m_name="$([[ $f =~ .*/(.*)[_].* ]] && echo ${BASH_REMATCH[1]})";
        find "$assets_path/media/marquees/" -maxdepth 1 -name "${name}_*.ppm" -exec rm -f {} \;
        cp -f "$m_path" "$assets_path/media/marquees/"
      done
    fi

    for f in .list .shooters .spinners; do
      [ -f "$temp_dir/$f" ] && { 7z a "$DIR/.backup/$arc_name" "$assets_path/GAMES/$f"; cp -fv "$temp_dir/$f" "$assets_path/GAMES/$f"; } | log_append_silent
    done
    for f in table timers clearram; do
      [ -f "$temp_dir/cafca_$f" ] && { 7z a "$DIR/.backup/$arc_name" "$cafca_dir/$f"; cp -fv "$temp_dir/cafca_$f" "$cafca_dir/$f"; } | log_append_silent
    done

    if [[ ${#new_pages[@]} -gt 0 ]]; then local path="$assets_path/.settings"
      declare -A page_titles=(); while read line; do line=${line//\"/}; page_titles["${line%% *}"]="${line#* }"; done< <(sed -n "/^page_titles=/ { s/^.*\x27\(.*\)\x27.*$/\1/; s/\x22//g; s/[=]/ /g; s/, /\n/g; p }" $path); page_keys=($(for k in "${!page_titles[@]}"; do printf "$k "; done | sed 's/ /\n/g' | sort))
      declare -a active_pages=($(awk -F'\x27' '{gsub(/(^\[ |\]$| \]$)/,"",$(NF-1)); n=split($(NF-1),pages,/ /); for (i in pages) if (length(pages[i]) > 0) print pages[i]; }' "$temp_dir/.list" | sort -u))
      for n in "${!new_pages[@]}"; do [[ ! "${page_keys[@]}" =~ "$n" ]] && [[ "${active_pages[@]}" =~ "$n" ]] && page_titles["$n"]="${new_pages[$n]}"; done; page_keys=($(for k in "${!page_titles[@]}"; do printf "$k "; done | sed 's/ /\n/g' | sort))
      output=$(for k in "${page_keys[@]}"; do printf "$k=\x22${page_titles[$k]}\x22, "; done | sed 's/, $//'); sudo sed -i "/^page_titles=/ s/=\([^#]*\)#/=\x27${output}\x27 #/" "$path"
      visible="$(sed -n "/^page_visible=/ { s/^.*\x27\(.*\)\x27.*$/\1/; p }" $path),$(echo "${!new_pages[@]}" | sed 's/ /,/g')"; sudo sed -i "/^page_visible=/ s/=\([^#]*\)#/=\x27${visible}\x27 #/" "$path"
    fi

    sudo rm -f $download_dir/*.* 2>/dev/null;
    find "$temp_dir/" -type f -exec rm -fv {} \; | log_append_silent
    printf '' > "$temp_dir/downloads"
    return 0
  fi

  return 1
}

# ___________________________________________________________________________________________________________________

downloader() {
  local items=()
  local log=
  local sample="$HOMEDIR/ADMIN/FILES/SAMPLES/success.wav"
  printf '' > "$temp_dir/downloads"

  echo " Starting transmission.."

  sudo -u pi bash -c 'transmission-daemon' &>/dev/null; sleep 1

  ( while [[ $(pgrep -fc 'transmission-daemon') -gt 0 ]]; do read -rsn1 a; read -rsn5 -t 0.01 b; case $a$b in $'\E') sudo pkill -f 'transmission-daemon'; break ;; esac; done & )

  declare -c id=$(sudo -u pi bash -c 'transmission-remote -l' | grep -A1 '^ID' | tail -n 1 | grep -m1 'MAME' | awk '{print $1}')
  [[ $id =~ ^[0-9]+$ ]] && sudo -u pi id=$id bash -c 'transmission-remote -t $id -r' &>/dev/null

  sudo -u pi file=$torrent_file bash -c 'transmission-remote -a $file; while [[ $(transmission-remote -l | grep -c Verifying) -gt 0 ]]; do :; done' &>/dev/null
  declare -c id=$(sudo -u pi bash -c 'transmission-remote -l' | grep -A1 '^ID' | tail -n 1 | grep -m1 'MAME' | awk '{print $1}')

  sudo -u pi id=$id bash -c 'transmission-remote -t $id -Gall' &>/dev/null
  sudo -u pi id=$id bash -c 'transmission-remote -t $id -f' | grep -we 'roms' | sed 's/:.*\/roms\// /' | sudo tee "$temp_dir/torrent.dump" &>/dev/null

  sed -i '/^.*.chd/d;/^.*.png/d' "$temp_dir/torrent.dump"

  local entries=($(awk '{ print $1"_"$2; }' "$temp_dir/torrent.dump")); for item in "${entries[@]}"; do items+=("${item%%_*}" "${item##*_}" OFF); done
  local indexes=$(whiptail --checklist "" --title "DOWNLOADER" 0 0 0 "${items[@]}" 2>&1 1>$(tty)); indexes=($(echo "$indexes" | sed 's/\x22//g'))
  local file_list=$(IFS=,; echo "${indexes[*]}")
  local printed=0; local pipe="$log_file"

  if [[ ${#indexes[@]} -gt 0 ]]; then
    [ -d $output_dir ] && sudo rm $output_dir/* 2>/dev/null

    sudo -u pi id=$id list=$file_list bash -c 'transmission-remote -t $id -g$list; transmission-remote -t $id --start' &>/dev/null

    while true; do
      log=$(sudo -u pi id=$id bash -c 'transmission-remote -t $id -f' | awk '{if ($4 ~ "Yes") print;}');
      for p in $(echo "$log" | awk '{gsub(/%/,"", $2);print $2}'); do [[ $p -lt 100 ]] && { downloading=1; break; }; done; clear; echo -e "Downloading:\n"
      [[ $printed -eq 0 ]] && printed=1 || pipe=/dev/null
      echo "$log" | awk 'NR==FNR{i=split($NF,x,/\/|\./);n=x[i-1];s[n]=$1"\t"$2"\t"n;l[n];next}{for (k in l) if ($2 == k){if(k in f == 0){$1="";$2="";gsub(/\x27/,"",$0);sub(/^\ +/,"",$0);s[k]=s[k]"\t\x27"$0"\x27"}f[k];next}}END{for(k in l)print s[k];}' - "$DIR/assets/mame2003-index" | column -t | awk -F'\x27' 'BEGIN{FS=OFS="\x27"}{gsub(/\ +/, " ",$(NF-1)); print}' | sudo tee -a $pipe 2>&1; sleep 1
      [[ $(echo "$log" | grep -wc '\s100%' ) -eq ${#indexes[@]} ]] && break
    done
  else
    return 1
  fi

  while [[ $(find $download_dir -type f -iname '*.part' | wc -l) -gt 0 ]]; do sleep 1; done; sleep 3

  [[ $play_sound -eq 1 ]] && sudo -u pi file="$sample" bash -c "aplay --quiet $file &>/dev/null" &

  find $download_dir -type f -iname '*.zip' ! -iname '*.part' -exec cp -v {} $roms_dir \; | log_append_silent

  for dir in $roms_dir; do sudo chown -R pi:pi $dir; sudo chmod -R 775 $dir; done
  for rom in $(find $download_dir -type f -iname '*.zip' ! -iname '*.part' -print | grep -o '[^/]*$'); do [ -f "$roms_dir/$rom" ] && echo "$roms_dir/$rom"; done | sudo tee "$temp_dir/downloads" &>/dev/null

  return 0
}

# -------------------------------------------------------------------------------

scraper() {
  local SCRAPER='/opt/retropie/supplementary/skyscraper/Skyscraper'
  local WORKDIR="$temp_dir/scraper"
  local gamelist="$WORKDIR/gamelist.xml"
  local writing=0
  local count=0

  mkdir "$WORKDIR" &>/dev/null; sudo cp "$HOMEDIR/.emulationstation/gamelists/mame-libretro/gamelist.xml" "$WORKDIR/gamelist.xml"

  [ ! -f "$temp_dir/.list" ] && cp -f "$assets_path/GAMES/.list" "$temp_dir/.list"; total=$(cat "$temp_dir/.list" | wc -l);

  if [[ $(cat "$temp_dir/downloads" | wc -l) -lt 1 ]]; then
    whiptail --yesno "SCRAPE ROMS DIR?" --title "NO DOWNLOADS" 0 0 0 2>&1 1>$(tty); [ $? -ne 0 ] && return 1
    find $roms_dir -type f -iname '*.zip' -exec bash -c '[[ ! ${1} =~ ${0} ]] && echo "${0}"' {} "$(awk -F'\x27' '{print $12}' $temp_dir/.list)" \; | sudo tee "$temp_dir/downloads" &>/dev/null
    #[ $? -eq 0 ] && find $roms_dir -type f -iname '*.zip' -exec bash -c '[[ ! ${1} =~ ${0} ]] && echo "${0}"' {} "$(awk -F'\x27' '{print $12}' $temp_dir/.list)" \; | sudo tee "$temp_dir/downloads" &>/dev/null
  fi

  if [[ $(cat "$temp_dir/downloads" | wc -l) -gt 0 ]]; then
    while read line; do line=$(echo "$line" | sed -E "s/\x1B\[([0-9]{1,2}(;[0-9]{1,2})*)?[m,K,H,f,J]//gm" | sed -E "s/\x1B\([A-Z]{1}(\x1B\[[m,K,H,f,J])?//gm");
      [[ "$line" =~ 'Pass 1' ]] && { writing=1; clear; echo "" | log_append ; } #sudo tee -a "$log_file"; }
      [[ $writing -eq 1 ]] && { (( count++ )); echo "$line" | log_append ; } #sudo tee -a "$log_file"; }
      [[ $count -ge 15 ]] && { writing=0; count=0; }
    done< <( $SCRAPER -p mame-libretro -s screenscraper -i "$roms_dir" -g "$WORKDIR" -a "$HOMEDIR/.skyscraper/artwork.xml" --flags unattendskip --includefrom "$temp_dir/downloads" ); echo -e "\nWRITING GAMELIST\n"

    ( while [[ $(pgrep -fc 'Skyscraper') -gt 0 ]]; do read -rsn1 a; read -rsn5 -t 0.01 b; case $a$b in $'\E') sudo pkill -f 'Skyscraper'; break ;; esac; done & )

    $SCRAPER -p mame-libretro -i "$HOMEDIR/RetroPie/roms/mame-libretro" -g "$WORKDIR" -a "$HOMEDIR/.skyscraper/artwork.xml" --flags unattend &>/dev/null

    local game_count=$(grep -Ec '<game>' "$WORKDIR/gamelist.xml")

    for (( n=1; n<=$game_count; n++ )); do
      data=$(awk -v p="<game>" -v n="$n" '$0~p{i++}i==n' "$WORKDIR/gamelist.xml")
      path=$(echo "$data" | grep -E "<path>" | awk '{$1=$1;print}' | sed 's/<path>//g; s/<\/path>//g')
      name=$(echo "$path" | grep -o '[^/]*$' | cut -d. -f1)

      if [[ $(grep -c "$name" "$temp_dir/downloads") -gt 0 ]] && [[ ${#data} -gt 0 ]]; then
        title=$(echo "$data" | grep -E "<name>" | awk '{$1=$1;print}' | sed 's/<name>//g; s/<\/name>//g')
        date=$(echo "$data" | grep -E "<releasedate>" | awk '{$1=$1;print}' | sed 's/<releasedate>//g; s/<\/releasedate>//g'); date="${date:0:4}"; [[ ${#date} -lt 1 ]] && date='19xx'
        publisher=$(echo "$data" | grep -E "<publisher>" | awk '{$1=$1;print}' | sed 's/<publisher>//g; s/<\/publisher>//g'); publisher=$(echo "$publisher" | cut -d, -f1); [[ ${#publisher} -lt 1 ]] && publisher='<unknown>'
        type='JOY'
        find "$assets_path/media/marquees" -type f -iname "$name*" -exec false {} +; marquee=$?
        echo "'$name' '$title' '$date' '$publisher' '$type' '$path' '$marquee' '[ ]'"

        [[ $(grep -c "<path>$path</path>" "$WORKDIR/gamelist.xml") -lt 1 ]] && echo "${data}" | sed 's/^/\t/' | sudo tee -a "$WORKDIR/gamelist.xml" &>/dev/null
      fi
    done | sudo tee "$WORKDIR/new_entries" &>/dev/null

    cat "$WORKDIR/new_entries" | awk '{print "0 "$0}' | sudo tee -a "$temp_dir/.list" &>/dev/null
    #cat "$temp_dir/.list" | sort -k4,4 -t \' --output "$temp_dir/.list"
    #awk '{$1="";print NR""$0}' "$temp_dir/.list" | sudo tee "$temp_dir/.list" &>/dev/null
    sort -k4,4 -t \' "$temp_dir/.list" --output "$temp_dir/.list"
    awk '{$1="";print NR""$0}' "$temp_dir/.list" > "$temp_dir/.list.new" && cp -f "$temp_dir/.list.new" "$temp_dir/.list"
    printf '' > "$temp_dir/downloads"

    whiptail  --msgbox "$(echo -e "FINISHED SCRAPING $(cat $WORKDIR/new_entries | wc -l) ROM(S)")" --title "SUCCESS" 0 0 2>&1 1>$(tty)
    return 0
  else
    whiptail  --msgbox "NOTHING TO SCRAPE.." --title "WARNING" 0 0 2>&1 1>$(tty)
    return 1
  fi
}

# ----------------------------------------------------------------------------------------------

update_lists() {
  echo -e "\nUPDATING LISTS..\n" | log_append

  # SHOOTERS
  echo -e "SHOOTERS:" | log_append_silent
  [ ! -f "$temp_dir/.shooters" ] && cp -f "$assets_path/GAMES/.shooters" "$temp_dir/.shooters"
  for game in $(awk -F'\x27' '{if (($10 ~ "GUN") || ($16 ~ "GUN")) print $2;}' "$temp_dir/.list"); do
    [[ $(grep -wc "^$game" "$temp_dir/.shooters") -eq 0 ]] && printf '%-8s\t1\n' "$game"
  done | sudo tee -a "$temp_dir/.shooters" | log_append_silent
  for game in $(grep -v '^$\|^\s*#' "$temp_dir/.shooters" | awk '{print $1}'); do
    [[ $(grep -c "$game.zip" "$temp_dir/.list") -lt 1 ]] && sed -i "/^${game}\s.*$/d" "$temp_dir/.shooters"
  done
  sort -k1,1 "$temp_dir/.shooters" --output="$temp_dir/.shooters"

  # SPINNERS
  echo -e "SPINNERS:" | log_append_silent
  [ ! -f "$temp_dir/.spinners" ] && sudo cp "$assets_path/GAMES/.spinners" "$temp_dir/.spinners"
  for game in $(awk -F'\x27' '{if (($10 ~ "SPIN") || ($16 ~ "SPIN")) print $2;}' "$temp_dir/.list"); do
    [[ $(grep -wc "^$game" "$temp_dir/.spinners") -eq 0 ]] && printf '%-8s\t72\t\t1\t\t255\t\t255\n' "$game"
  done | sudo tee -a "$temp_dir/.spinners" | log_append_silent
  for game in $(grep -v '^$\|^\s*#' "$temp_dir/.spinners" | awk '{print $1}'); do
    [[ $(grep -c "$game.zip" "$temp_dir/.list") -lt 1 ]] && sed -i "/^${game}\s.*$/d" "$temp_dir/.spinners"
  done
  sort -k1,1 "$temp_dir/.spinners" --output="$temp_dir/.spinners"

  # CAFCA
  echo -e "\nCAFCA FILES:" | log_append_silent
  for file in table timers clearram; do [ ! -f "$temp_dir/cafca_$file" ] && cp -f "$cafca_dir/$file" "$temp_dir/cafca_$file"; echo -e "\n${file^^}:" | log_append
    for game in $(awk -F'\x27' '{print $2}' "$temp_dir/.list"); do
      [[ $(grep -wc "^$game" "$temp_dir/cafca_$file") -eq 0 ]] && echo "$game $([[ $file == timers ]] && echo 1 || { [[ $file == clearram ]] && echo 0; })"
    done | sudo tee -a "$temp_dir/cafca_$file" &>/dev/null
    sort -k 1,1 "$temp_dir/cafca_$file" | column -t | sudo tee "$temp_dir/cafca_$file" &>/dev/null
  done

  return 0
}

# --------------------------------------------------------------------------------------

format_table() { local cmd=''
  [ $# -lt 1 ] && { echo ""; return; }
  [ -f "$1" ] && cmd='cat "$1"' || cmd='echo "$1"'
  [ ! -f "$temp_dir/.list" ] && cp -f "$assets_path/GAMES/.list" "$temp_dir/.list"

  for file in table timers clearram; do [ ! -f "$temp_dir/cafca_$file" ] && cp -f "$cafca_dir/$file" "$temp_dir/cafca_$file"; done

  eval "$cmd" | awk -F '\x27' -v path="$temp_dir" '{
    n=split($4,t,/\ -\ |\(|\//); $4=t[1]; if(length($4)>8){$4=substr($4,1,6);$4=$4".."}gsub(/\ +/," ",$16);gsub(/\[|\]/,"",$16);sub(/^\ /,"",$16);n=split($16,s," ");for(i=0;i<=n;i++)page=page""substr(s[i],1,1);gsub(/^\ +|\ +$/,"",page)
    param="table";    cmd="grep -w  "$2" "path"/cafca_"param; cmd | getline str; close(cmd); gsub(/\ +/," ",str); n=split(str,d," "); byte=((length(d[2])>0?"x":"-")); sub(/^0x/,"",byte); key=substr(param,1,3); data[key]=byte
    param="clearram"; cmd="grep -w  "$2" "path"/cafca_"param; cmd | getline str; close(cmd); gsub(/\ +/," ",str); n=split(str,d," "); byte=((length(d[2])>0?d[2]:"0")); key=substr(param,1,3); data[key]=byte
    param="timers";   cmd="grep -w  "$2" "path"/cafca_"param; cmd | getline str; close(cmd); gsub(/\ +/," ",str); n=split(str,d," "); byte=((length(d[2])>0?d[2]:"0")); key=substr(param,1,3); data[key]=byte
    printf("%-4s %-9s %-8s %-4s %-2s [ %-3s ] %-1s %-1s %-1s %-2s\n",$1,$2,$4,$6,substr($10,1,3),page,data["tab"],$14,data["cle"],data["tim"]); page=""
  }'
}

# ----------------------------------------------------------------------------------------------

fbConvert() {
  [ $# -lt 2 ] && { echo "Error: not enough args!" | append_log; return 1; }
  local img="$1"
  local output="$2"
  sudo fbi -T 2 -d /dev/fb0 --once --noverbose "$img" &>/dev/null
  sleep 1
  cat /dev/fb0 > "$output"
  while [[ $(pgrep -c fbi) -gt 0 ]]; do :; done;
}

# ----------------------------------------------------------------------------------------------

prog() {
  [ $1 -eq 0 ] && { p_max=$2; return; }

  local txt="${2:-Please wait..}"
  local p="$(( (( $1 * 1000) / p_max) / 10 ))"

  if [[ $(( p % 10 )) -eq 0 ]] && [[ $p_prnt -eq 0 ]]; then p_prnt=1
    sudo -u pi p="$p" txt="$txt\n( $1 / $p_max )\n" bash -c 'echo "$p" | dialog --colors --gauge "$txt" 10 $(( $(tput cols)/2 )) 0' #2>&1 1>$(tty)
    [[ $p -eq 100 ]] && { sleep 0.5; clear; }
  elif [[ $(( p % 10 )) -ne 0 ]] && [[ $p_prnt -eq 1 ]]; then
    p_prnt=0
  fi
}

# -------------------------------------------------------------------------------

checkArgs() {
  [[ ${#args[@]} -lt 1 ]] && return 1
  [[ ! -f "$temp_dir/.list" ]] && cp -f "$assets_path/GAMES/.list" "$temp_dir/.list"

  local thd_mode="$(sed -n '/THD_MODE=/ { s/\(.*\)=//g;s/\x27//g;p }' $HOMEDIR/ADMIN/.sys/.states)"
  local action="${args[0]}"
  local total=$(cat "$temp_dir/.list" | wc -l)
  local exit_code=0

  #[[ ! $thd_mode == admin ]] && { sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --mode admin; sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --grab joystick; }

  case "${action,,}" in
    'edit')
      #[[ "${args[1]}" =~ ^[0-9]+$ ]] && [[ ${args[1]} -le $total ]] && { Edit "${args[1]}"; [ $? -eq 1 ] && Apply; exit $?; }
      if [[ "${args[1]}" =~ ^[0-9]+$ ]] && [[ ${args[1]} -le $total ]]; then
        Edit "${args[1]}"; exit_code=$?
        if [[ $exit_code -eq 1 ]]; then
          Apply; exit_code=$?
         #[[ $? -eq 0 ]] && [[ ${#new_pages[@]} -gt 0 ]] && exit_code=2
        fi
      fi
      exit $exit_code
      ;;
    'add')
      [[ ${#args[1]} -gt 0 ]] && Append "${args[1]}" || Append
      ;;
  esac

  #[[ ! $thd_mode == admin ]] && { sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --mode "$thd_mode"; }

  clear
  exit 1
}

# ______________________________________________________________________________________________


cleanup() { sudo /etc/init.d/transmission-daemon stop; sudo pkill -f "transmission-daemon"; echo -en "\033[9;${t_stored:-2}]" > /dev/tty1; exit 0; }

exit_code=0
declare -a global_funcs=()
declare -A new_pages=()

while IFS=, read -ra globals; do for g in "${globals[@]}"; do [[ "$g" =~ ^'f '.* ]] && global_funcs+=("${g##* }"); done; done< <({ declare -p; declare -F; } | grep '^declare' | grep -Ev '\s[A-Z_]+' | sed 's/[=].*$//; s/^declare\s-//g; s/^ //; s/$/,/')

checkArgs
trap 'cleanup &>/dev/null' EXIT INT HUP SIGINT SIGHUP TERM

while [[ $exit_code -eq 0 ]]; do Main; exit_code=$?; done


#if [[ $(cat "$temp_dir/downloads" | wc -l) -gt 0 ]] && [[ ! -f "$temp_dir/.list" ]]; then
#  whiptail --yesno "$(cat tmp/downloads | grep -o '[^/]*$')\n" --title "DELETE ROMS?" 0 0 0 2>&1 1>$(tty)
#  [ $? -eq 0 ] && { clear; for rom in $(cat "$temp_dir/downloads"); do sudo rm -vf "$rom"; done | log_append ; }
#fi

exit 0

# ----------------------------------------------------------------------------------------------------------------------------------------------
