#!/bin/bash

ARG="$@"; PID=$$; PS1_STORED=$PS1

UNMARK="$(echo -e "\e[0m")"
  MARK="$(echo -e "\e[7m")"
    NC="$(echo -e "\033[0m")"
     C="$(echo -e "\033[32m")"

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# PATHS

SCRIPT=$(readlink -f "$0")
DIR=$(cd "$(dirname $SCRIPT)" && pwd)
HOMEDIR=$(getent passwd pi | cut -d: -f6)

assets_path="$DIR/.assets"
media_path="$assets_path/media"
list_path="$assets_path/GAMES/.list.tmp"
font_path="$assets_path/fonts"
cafca_path="$HOMEDIR/CAFCA/data/mame-libretro"

serial_path=$(sed -n '/fifo_com/ { s/\(.*\)=//g; s/\x27//g; p }' "$HOMEDIR/ADMIN/.sys/.settings")
roms_path=$(sed -n '/dir_roms/ { s/\(.*\)=//g; s/\x27//g; p }' "$HOMEDIR/ADMIN/.sys/.settings")
gun_serial=$(sed -n '/serial_gun=/ { s/\(.*\)=//g; s/\x27//g; s/ //; p }' "$HOMEDIR/ADMIN/.sys/.settings" | tail -c 2)

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

console=$(sed -n '/console=/ { s/\(.*\)=//g; s/\x27//g; p }' "$HOMEDIR/ADMIN/.sys/.settings")
prev_state=$(sed -n '/UI=/ { s/\(.*\)=//g; s/\x27//g; p }' "$HOMEDIR/ADMIN/.sys/.states")
width=$(echo "$(tput cols)"); [[ $(( width % 2 )) > 0 ]] && width=$(( width - 1 ))
height=$(echo "$(tput lines)"); maxPage=$(( height - 3 )); maxPage_stored="$maxPage"

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

Init() {
  declare -g keyhelp=$(cat "$assets_path/GAMES/keyhelp")
  declare -g blanktime=$(cat /sys/module/kernel/parameters/consoleblank)
  declare -g spinner_off=$(sed -n '/disable_spinner/ { s/\(.*\)=//g; s/\x27//g; p }' "$HOMEDIR/ADMIN/.sys/.settings")
  declare -g footer_msg=''

  loadSettings

  for arr in list indexes offsets headers; do declare -ag "${arr}"; done
  for var in navigated idle_count pageIndex pageIndexPrev page offset stored_offset stored_active active prev vpos mod_count total_files header_height page_changed footer_msg marquee_on; do declare -g "${var}"=0; done

  createPages; [[ -z "$page" ]] && page="${pages[$pageIndex]}"

  [[ ! -z "$sorting_table" ]] && { declare -Ag sortings=(); for sort in `echo ${sorting_table}`; do sortings["${sort%%=*}"]="${sort##*=}"; done; }
  [[ $remap -eq 1 ]] && { declare -Ag remaps=(); for rmp in `echo ${remapping}`; do remaps["${rmp%%=*}"]="${rmp##*=}"; done; }
  [[ $hide_term -eq 1 ]] && { PS1=; stty -F /dev/tty1 -echo; sudo setterm --cursor off; }

  [ -f "$assets_path/fb0.dump" ] && { sudo rm "$assets_path/fb0.dump"; }
  [ -z "$NEWT_COLORS" ] && export NEWT_COLORS="$(cat $assets_path/config/whiptail_theme.cfg)"

  trap 'exitFunc' EXIT; trap 'exit 1' INT HUP SIGINT SIGHUP TERM

  sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --mode menu
  sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --grab joystick

  [[ $spinner_off -eq 0 ]] && { sendSerial "SPIN_OFF 0"; initSpin; }
  [[ $kiosk_mode -eq 1 ]] && admin_mode=0

  sed -i "/ACTIVE_GAME/ s/[=].*$/=\'\'/" "$HOMEDIR/CAFCA/data/.settings"
  sed -i "/UI=/ s/[=].*$/=\x27GAMES\x27/" "$HOMEDIR/ADMIN/.sys/.states"
  sendSerial "$( (( test_mode )) && echo 'VFD 0 EDIT MODE' || echo 'VFD 5     G A M E S' )"
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

Main() {
  Init
  setPage; showOrder

  while true; do
    input=$(readInput)

    if [[ ${#input} -gt 0 ]]; then
      navigate "${input}"
    else (( idle_count++ ));
      [[ $idle_count -gt "$(( t_title * 10 ))" ]] && showTitle
    fi
  done
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

navigate() {
  local key="${@}"

  if [[ $test_input -eq 1 ]]; then
    clear; echo -e "\n  $key -> $key ( ${remaps[$key]} )"
    [[ "$key" == ESC ]] && { sleep 1; exit; }
  else
    [[ $remap -eq 1 ]] && [[ ! -z "${remaps[$key]}" ]] && key="${remaps[$key]}"

    if [[ ! "$key" == MOD ]]; then
      [[ $mod_count -ge 3 ]] && { showHelp 0; return; } || mod_count=0
      case "${key}" in 'D')         ;; *) [[ $marquee_on -eq 1 ]] && showMarquee 0 ;; esac
      case "${key}" in 'U'|'L'|'R') ;; *) [[ $show_footer -eq 1 ]] && [[ ${#footer_msg} -gt 0 ]] && footer ;; esac
      case "${key}" in
        'UP')     prev=$active; (( active-- )); checkList ;;
        'DOWN')   prev=$active; (( active++ )); checkList ;;
        'LEFT')   [[ $kiosk_mode -ne 1 ]] && { pageIndexPrev=$pageIndex; (( pageIndex-- )); changePage; showOrder; } ;;
        'RIGHT')  [[ $kiosk_mode -ne 1 ]] && { pageIndexPrev=$pageIndex; (( pageIndex++ )); changePage; showOrder; } ;;
        'PGUP')   [[ ${#offsets[@]} -gt 1 ]] && { (( offset-- )); makeList; drawScreen; showOrder; } ;;
        'PGDOWN') [[ ${#offsets[@]} -gt 1 ]] && { (( offset++ )); makeList; drawScreen; showOrder; } ;;
        'U') 	  toggleFav 	;;
        'D') 	  showMarquee 1 ;;
        'L') 	  setupUI 	;;
        'R')      editEntry     ;;
        'ENTER')  launchGame 	;;
        'START')  launchGame 	;;
        'ESC')    quitMenu 	;;
      esac
      case "${key}" in 'UP'|'DOWN'|'PGUP'|'PGDOWN') navigated=1; title_shown=0; idle_count=0 ;; *) idle_count=0 ;; esac
    else (( mod_count++ ))
      [[ $mod_count -eq 3 ]] && showHelp 1
    fi
  fi
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

launchGame() {
  local entry=$(( ${offsets[$offset]} + active )); idx=${indexes[$entry]}
  local path=$(sed "${idx}q;d" "$list_path" | cut -d\' -f12)
  local type=$(sed "${idx}q;d" "$list_path" | cut -d\' -f10)

  if [ -f "$path" ]; then
    local system=$(echo "$path" | cut -d/ -f6)
    local game=$(echo "$path" | grep -o '[^/]*$' | cut -d. -f1)
    clear

    if [[ $type == SPIN ]] && [[ $spinner_off -eq 0 ]]; then sendSerial "SPIN_OFF 0"
      [[ $(grep -c "$game" "$assets_path/GAMES/.spinners") -eq 0 ]] && {  sendSerial "SPIN_SENS 72"; sendSerial "SPIN_SPEED 1"; } || ( while read val; do sendSerial "$val"; sleep 1; done< <(grep -w -m1 "^$game" "$assets_path/GAMES/.spinners" | awk '{print "SPIN_SENS "$2; print "SPIN_SPEED "$3}') ) &
    elif [[ $type == GUN ]]; then fire_mode=
      [[ $(grep -wc "$game" "$assets_path/GAMES/.shooters") -gt 0 ]] && { fire_mode=$(grep -w -m1 "$game" "$assets_path/GAMES/.shooters" | awk '{print $2}'); } #sendSerial "${gun_serial:-0}:0 ${fire_mode:-1}"
    fi

    ( sleep 5; showTitle ) &

    # - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    # START GAME

    sudo sed -i "/ACTIVE_GAME/ s/[=].*$/=\'${system}\/${game}\'/" "$HOMEDIR/CAFCA/data/.settings"
    sudo sed -i "/UI=/ s/[=].*$/=\x27retroarch\x27/" "$HOMEDIR/ADMIN/.sys/.states"

    sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --mode game
    sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --remove joystick
    sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --remove spinner

    setBlankTime 0
    sudo cat /dev/fb0 > "$assets_path/fb0.dump" & wait $!

    # - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    sudo -u pi bash -c "python3 /opt/retropie/configs/all/CRT/bin/GeneralModule/emulator_launcher.py ${path} ${system} dummy"

    # - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    # END GAME

    sudo sed -i "/ACTIVE_GAME/ s/[=].*$/=\'\'/" "$HOMEDIR/CAFCA/data/.settings"
    sudo sed -i "/UI=/ s/[=].*$/=\x27GAMES\x27/" "$HOMEDIR/ADMIN/.sys/.states"

    sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --mode menu
    sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --grab joystick
    initSpin

    sendSerial "${gun_serial:-0}:0 1"

    [[ "${system}" == mame-libretro ]] && checkCfg "${game}"
    [ -f "$assets_path/fb0.dump" ] && { sudo cat "$assets_path/fb0.dump" > /dev/fb0; sudo rm "$assets_path/fb0.dump"; }

    sudo setterm --cursor off; Redraw
    navigated=0; title_shown=1; page_changed=0
  fi
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

makeList() {
  declare -ag screen_data=()
  local data=''
  local line=''

  clear

  [[ $show_header -eq 1 ]] && header

  offset=$(( offset % ${#offsets[@]} )); list=(); screen_data=(); active=0

  for x in $(seq ${offsets[$offset]} $(( (${offsets[$offset]} + maxPage)-1 ))); do
    [[ $x -ge $total_files ]] && break;

    if [[ $show_data -eq 1 ]]; then
      data=$(awk -F'\x27' -v l="${indexes[$x]}" -v tbl="$cafca_path/table" 'NR==l{cmd="grep -wE \"^"$2"\" "tbl; cmd | getline a; close(cmd); gsub(/^[a-z] */,"",a); a=(length(a)>0)?1:0; b=($16~/FAV/)?1:0; i=$10" "$(NF-1); t=(match(i,/(GUN|SPIN|CAR)/))?substr(i,RSTART,1):"J"; print a,$14,b,t}' $list_path)
    else
      if [[ $page == @(70S|80S|90S) ]] || [[ $sorting == AGE ]]; then
        data=$(awk -F'\x27' -v l="${indexes[$x]}" 'NR==l{print $6}' $list_path)
      elif [[ $page == TOP ]] || [[ $sorting == TYPE ]]; then
        data=$(awk -F'\x27' -v l="${indexes[$x]}" 'NR==l{print substr($10,1,1)}' $list_path)
      fi
    fi

    title=$(awk -F'\x27' -v l="${indexes[$x]}" -v w="$((width-line_limit))" 'NR==l { if ((length($4) > w) && ($4 ~ / - /)) gsub(/ - [^(\[]*[^(\[\$]/,"",$4); print $4 }' $list_path)
    list+=("$title"); screen_data+=("$data")
  done

  for i in $(seq 0 $(( ${#list[@]}-1 ))); do printer "${list[$i]}" 0 "${screen_data[$i]}"; done

  [[ $show_footer -eq 1 ]] && footer
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

checkList() {
  if [[ $active -lt 0 ]]; then
    [[ ${#offsets[@]} -gt 1 ]] && { (( offset-- )); makeList; active=$((${#list[@]}-1)); } || { active=0; }
  elif [[ $active -ge ${#list[@]} ]]; then
    [[ ${#offsets[@]} -gt 1 ]] && { (( offset++ )); makeList; active=0; } || { active=$((${#list[@]}-1)); }
  fi
  drawScreen
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

drawScreen() {
  tput sc; tput cup $(( vpos + prev )) 0
  printer "${list[$prev]}" 0 "${screen_data[$prev]}"; tput rc
  tput sc; tput cup $(( vpos + active )) 0
  printer "${list[$active]}" 1 "${screen_data[$active]}"; tput rc
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

Redraw() { stored_active=$active; makeList; prev=0; active=$stored_active; drawScreen; }

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

setPage() {
  local rdrw="${1:-1}"
  local s="${sortings[${sorting:-1}]:-1}"

  indexes=($(awk -F'\x27' -v n="$s" -v pg="${page/ALL/ }" -v o_t="$order_types" 'BEGIN{split(o_t,t,/[,]/); for (o in t) types[t[o]]=o;}{ gsub(/ /,"",$1); p=substr(pg,1,1); val=$(n); if ((p~/7|8|9/) && ($6 ~ "^(19"p((p==9)?"|200)":")")) || $(NF-1)~pg ){ val=(val~/19xx/)?"2050":val; val=(n==10)?types[val]:val; if (n == 1) print $1; else str=val"_"$1","str;}} END {if (n != 1) system("printf " str " | tr \",\" \"\\n\" | sort | cut -d_ -f2");}' "$list_path"))
  total_files=${#indexes[@]}; offsets=($(for x in $(seq 0 $maxPage "$(( ${#indexes[@]}-1 ))"); do echo $x; done)); offset=0

  [[ $rdrw -eq 1 ]] && { makeList; drawScreen; }
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

changePage() {
  pageIndex=$(( pageIndex % ${#pages[@]} )); page="${pages[$pageIndex]}"
  setPage; page_changed=1; navigated=0
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

createPages() {
  awk -F'\x27' 'BEGIN{FS=OFS="\x27"}{if ($10~/(GUN|SPIN|CAR)/ && $(NF-1)~/(FAV|TOP)/) gsub(/\]$/,$10" ]",$(NF-1)); print; }' "$assets_path/GAMES/.list" | sudo tee "$list_path" &>/dev/null

  declare -ag active_pages=($(awk -F'\x27' '{gsub(/(^\[ |\]$| \]$)/,"",$(NF-1)); gsub(/^ $/,"ALL",$(NF-1)); n=split($(NF-1),p,/ /); for (i in p) print p[i]; dec=substr($6,3,1); if (dec~/(7|8|9)/) print dec"0S"; }' "$list_path" | sort -u | sort))
  declare -ag pages_visible=($(sed -n '/# GLOBAL\|# GAMES/,/# EOF_GLOBAL\|# EOF_GAMES/ { s/#.*$//; /^\s*$/d; /^page_visible=/ { s/^.*\x27\(.*\)\x27.*$/\1/; s/[,]/ /g; p }}' "$assets_path/.settings"))
  declare -ag pages=()
  declare -Ag pageTitles=()

  for p in "${active_pages[@]}"; do [[ "${pages_visible[@]}" =~ "$p" ]] && { pages+=("$p"); pageTitles["$p"]="$([[ ${p:0:1} == @(7|8|9) ]] && echo "19${p,,}" || echo "$p")"; }; done
  while read x; do x=${x//\"/}; [[ ${pages_visible[@]} =~ ${x%% *} ]] && pageTitles["${x%% *}"]="${x#* }"; done< <(sed -n '/^page_titles=/ { s/^.*\x27\(.*\)\x27.*$/\1/; s/\x22//g; s/[=]/ /g; s/, /\n/g; p }' "$assets_path/.settings");
  for p in $(seq 0 $(( ${#pages[@]}-1))); do [[ "${pages[$p]}" == "${start_page}" ]] && { pageIndex=$p; page="${pages[$pageIndex]}"; }; done

  createHeaders
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

createHeaders() {
  local font=''; local idx=; local key=; local str=; local txt=; local w=$(( width - 2 ));

  declare -ag fonts=('halfiwi' 'Straight' 'Small' 'Mini' 'kompaktblk' 'smslant' 'Chunky' 'ublk' 'Stampatello' 'eftifont')
  declare -Ag font_ids=( [70S]="0 1 2 4 7" [80S]="0 1 2 5 8" [90S]="0 1 3 6 9" ); declare -Ag font_idx=( [70S]="0" [80S]="0" [90S]="0" )
  declare -ag uniq_idx=($(for i in $(seq 1 "$(echo ${font_ids[@]:0:1} | wc -w)"); do echo $i; done | shuf))

  makeArr "${!font_ids[@]}";

  headers=()

  for (( p=0; p<${#pages[@]}; p++ )); do font='Calvin_S'; idx=0; key="${pages[$p]}"
    [[ "${pages[$p]}" == @(70S|80S|90S) ]] && { [[ $random_fonts -gt 0 ]] && idx=$(echo "${font_ids[$key]}" | awk -v i="${font_idx[$key]}" '{print $(i)}'); font="${fonts[$idx]}"; }
    font="$font_path/$font.flf"; str=$(printf "${pageTitles[${pages[$p]}]}"); txt=$(printf "$(figlet -w $w -c ${str,,} -f $font)"); headers+=("$(printf "${txt}")")
    [[ $(echo "${headers[-1]}" | wc -l) -gt $header_height ]] && { header_height="$( echo "${headers[-1]}" | wc -l)"; }
  done

  maxPage=$(( maxPage_stored - 6 ));

  for (( h=0; h<${#headers[@]}; h++ )); do
    local hheight=$(echo "${headers[$h]}" | wc -l)
    local margin=$(( header_height - hheight ))
    [[ $margin -gt 0 ]] && for i in $(seq 1 $((margin-1)) ); do headers[$h]="\n${headers[$h]}"; done
  done
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

header() {
  local text=$(printf "${headers[$pageIndex]}"); [[ $colors -gt 0 ]] && text=${C}${text}${NC}
  local line=$(printf "$(for i in $(seq 0 $(( width-1 ))); do printf "${line_char:-.}"; done)"); [[ $colors -gt 0 ]] && line=${C}${line}${NC}

  tput cup 0 0; echo "${text}"; vpos=$header_height; tput cup $vpos 0; printf "${line}";
  [[ $show_data -eq 1 ]] && { tput cup $vpos $(( width-7 )); printf "${C}D M F T${NC}"; }
  (( vpos+=2 )); tput cup $vpos 0; tput sc
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

footer() {
  local msg="~ $total_files GAMES"
  local line=$(printf "$(for i in $(seq 0 $(( width-1 ))); do printf "${line_char:-.}"; done)")
  local info=$(echo "$(for x in $(seq 0 ${#offsets[@]});do [[ ${offsets[$x]} =~ ${offsets[$offset]} ]] && { echo $((x+1));break; };done) / ${#offsets[@]}")
  local spacing=$(( ${#info}+1 ))
  local bottom=$(( $(tput lines)-3 ))

  [[ $colors -gt 0 ]] && { info=${C}${info}${NC}; msg=${C}${msg}${NC}; line=${C}${line}${NC}; }

  tput sc
  tput cup $bottom 0; echo "${line}"

  [[ ${#footer_msg} -gt 0 ]] && { tput cup $((bottom+2)) 0; for i in $(seq 1 $width); do printf " "; done; footer_msg=''; }

  tput cup $((bottom+2)) 0; printf "${msg}"
  tput cup $((bottom+2)) $(( width-spacing )); printf "${info}";
  tput rc
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

showTitle() {
  local title=''

  if [[ $navigated -eq 1 ]]; then
    local i="${indexes[$((${offsets[$offset]}+active))]}"
    title=$(awk -F'\x27' -v l="$i" 'NR==l {gsub(/ - .*$| \(.*$/,"",$4);if($4 ~ /.*, The/){gsub(/, The/,"",$4);$4="The "$4} print $4" ("$6")","["$8"]"}' "$list_path")
    sendSerial "VFD 1 $title"
    navigated=0
  elif [[ $page_changed -eq 1 ]]; then
    title="${pageTitles[$page]}";
    sendSerial "VFD 4 $(printf '%*s' $(((16-${#title})/2)); printf '%s\n' "$title")"
    #sendSerial "VFD 1 _.\xb7\xab\xb0\xa4  $title  \xa4\xb0\xbb\xb7._"
  fi
  page_changed=0
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

showMarquee() {
  local entry=$(awk -v l="${indexes[$((${offsets[$offset]}+active))]}" 'NR==l {print}' "$list_path")
  local name=$(echo "$entry" | awk -F'\x27' '{print $2}')
  local img=$(echo "$entry" | awk -F'\x27' '{print $14}')

  local file=$(find "$media_path/marquees" -type f -iname "$name*.ppm" -print -quit)
  local HEIGHT=$(cat /sys/class/graphics/fb0/virtual_size | cut -d, -f2)
  local WIDTH=$(cat /sys/class/graphics/fb0/stride)

  if [[ $# -lt 1 ]] || [[ $1 -lt 1 ]]; then
    [ -f "$assets_path/fb0.dump" ] && { sudo cat "$assets_path/fb0.dump" > /dev/fb0; sudo rm "$assets_path/fb0.dump"; }
    marquee_on=0
  elif [[ $1 -gt 0 ]]; then
    if [ -f "$file" ] && [[ $img -gt 0 ]]; then
      H="${file##*/}"; H="${H/.*/}"; H="${H##*_}"; PAD=$(( (HEIGHT - H) / 2 ))
      sudo cat /dev/fb0 > "$assets_path/fb0.dump" & wait $!
      ( dd if="$file" bs=1 skip=$(( WIDTH * PAD )) count=$(( WIDTH * H )) iflag=skip_bytes,count_bytes | dd seek=$(( WIDTH * PAD )) oflag=seek_bytes of=/dev/fb0 ) >/dev/null 2>&1
      marquee_on=1
    else
      sendSerial "VFD 1 NO MARQUEE.."
      marquee_on=0
    fi
  fi
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

showFootMsg() {
  local bottom=$(( $(tput lines)-3 ))
  local msg="$@"; footer_msg="${msg}"

  [[ $colors -gt 0 ]] && { msg=${C}${msg}${NC}; }

  tput sc
  tput cup "$(( $(tput lines)-1 ))" "$(( (($(tput cols)/2)-(${#footer_msg}/2))-2 ))"
  printf "${msg}"
  tput rc
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

showOrder() {
  local s="${sortings[${sorting:-1}]:-1}"
  if [[ ! -z $sorting ]]; then
    [[ ! $sorting == NONE ]] && [[ ! $s -eq 1 ]] && { showFootMsg "ORDER: $sorting"; footer_msg=''; }
  fi
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

toggleData() {
  show_data="$(( ! $show_data ))"; Redraw
  [[ $show_footer -eq 1 ]] && showFootMsg "DATA $([[ $show_data -gt 0 ]] && echo ON || echo OFF)"
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

toggleAdmin() {
  local exit_code=0
  [[ $admin_mode -eq 0 ]] && { Password ${1:-0} 2>&1 1>/dev/tty; exit_code=$?; }

  if [ $exit_code -eq 0 ]; then admin_mode="$(( ! $admin_mode ))"
    local state=$(awk -v mode="$admin_mode" 'BEGIN{print(mode>0)?"ON":"OFF"}')
    echo "VFD $(((admin_mode>0)?6:3)) ADMIN MODE $state" > /tmp/serial.fifo
  elif [[ $exit_code -eq 1 ]]; then
   echo "VFD 6      WRONG PASS" > /tmp/serial.fifo
  fi

  return $exit_code
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

toggleFav() {
  local target="${1:-$start_page}"
  local action=
  local index="${indexes[$((${offsets[$offset]}+active))]}"
  local pges=$(awk -F '\x27' -v l="$index" 'NR==l {print $(NF-1)}' "$list_path" | sed 's/\[ *[^[:alnum:]]*//g;s/[^[:alnum:]]*\( *\|\]\)$//')
  local name=$(awk -F '\x27' -v l="$index" 'NR==l {print $2}' "$list_path")

  [[ $admin_mode -eq 0 ]] && { toggleAdmin; [ $? -ne 0 ] && return; }

  [[ ! "$pges" =~ "$target" ]] && { pges="$target $pges"; action=1; } || { pges="${pges/$target/}"; action=0; }
  pges=$(echo "[ $pges ]" | awk '{$1=$1;print}'); sed -i "${index}{s/'[^']*'/'${pges}'/8}" "$assets_path/GAMES/.list"
  awk -F'\x27' 'BEGIN{FS=OFS="\x27"}{if ($10~/(GUN|SPIN|CAR)/ && $(NF-1)~/(FAV|TOP)/) gsub(/\]$/,$10" ]",$(NF-1)); print; }' "$assets_path/GAMES/.list" | sudo tee "$list_path" &>/dev/null

  clear; sleep 0.5

  [[ $page == TOP ]] && setPage || Redraw
  [[ ! -z $action ]] && [[ $show_footer -eq 1 ]] && showFootMsg "$([[ $action -eq 1 ]] && echo "ADDED ${name^^} TO" || echo "REMOVED ${name^^} FROM") $target"
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

editEntry() {
  local index="${indexes[$((${offsets[$offset]}+active))]}"
  local name=$(awk -F '\x27' -v l="$index" 'NR==l {print $2}' "$list_path")
  local code=1

  [[ $admin_mode -eq 0 ]] && { toggleAdmin; [ $? -ne 0 ] && return; }

  declare -A pg_titles=([OLD]="$(sed -n '/^page_titles=/ { s/^.*\x27\(.*\)\x27.*$/\1/; p }' $assets_path/.settings)")
  declare -A pg_visibl=([OLD]="$(sed -n '/^page_visible=/ { s/^.*\x27\(.*\)\x27.*$/\1/; p }' $assets_path/.settings)")

  sudo bash "$assets_path/helpers/editor/editor.sh" edit $index; code=$?; sudo setterm --cursor off

  pg_titles[NEW]="$(sed -n '/^page_titles=/ { s/^.*\x27\(.*\)\x27.*$/\1/; p }' $assets_path/.settings)"
  pg_visibl[NEW]="$(sed -n '/^page_titles=/ { s/^.*\x27\(.*\)\x27.*$/\1/; p }' $assets_path/.settings)"


  if [[ ! "${pg_titles[NEW]}" == "${pg_titles[OLD]}" ]] || [[ ! "${pg_visibl[NEW]}" == "${pg_visibl[OLD]}" ]]; then
    pageIndexPrev=$pageIndex; stored_offset=$offset; createPages
    pageIndex=$pageIndexPrev; page="${pages[$pageIndex]}"; setPage 0; offset=$stored_offset
    #[[ $show_footer -eq 1 ]] && showFootMsg "$([[ $action -eq 1 ]] && echo "UPDATED ${name^^}!")"
  fi

  Redraw

  #createPages; setPage; [[ $show_footer -eq 1 ]] && showFootMsg "$([[ $action -eq 1 ]] && echo "UPDATED ${name^^}!")"
  #pageIndexPrev=$pageIndex; stored_active=$active; createPages; pageIndex=$pageIndexPrev; page="${pages[$pageIndex]}"; setPage 0; prev=0; active=$stored_active; drawScreen

  #echo "VFD 3 CODE: $code" > /tmp/serial.fifo
  #if [[ $code -eq 0 ]]; then
  #  createPages; setPage; [[ $show_footer -eq 1 ]] && showFootMsg "$([[ $action -eq 1 ]] && echo "UPDATED ${name^^}!")"
  #else
  #  Redraw
  #fi
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

showHelp() {
  local content="${keyhelp}"; [[ $colors -eq 1 ]] && content="${C}${keyhelp}${NC}"
  if [ $1 -eq 1 ]; then
    tput sc; clear; sleep 0.5; printf "$content"; tput cup 0 0
  else
    [[ $mod_count -ge 3 ]] && { tput rc; Redraw; }
    mod_count=0
  fi
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

setupUI() {
  [[ $admin_mode -eq 0 ]] && { toggleAdmin; [ $? -ne 0 ] && return; }

  declare -a ui_options=()
  local code=1
  local sel=$(whiptail --menu "" 0 0 0 "DATA VIEW" "($([[ $show_data -gt 0 ]] && echo ON || echo OFF))" "SORT ORDER" "($sorting)" 2>&1 1>$(tty)); code=$?

  sudo setterm --cursor off

  if [[ $code -eq 0 ]]; then
    case "$sel" in
      'DATA VIEW') toggleData; return ;;
      'SORT ORDER')
        for key in $(sed -n "/^sorting_table=/ { s/^.*\x27\(.*\)\x27.*$/\1/; s/=[[:digit:]]\+//g; p }" "$assets_path/.settings" | sort); do ui_options+=("$key" ""); done
        val=$(whiptail --title "ORDER" --menu "" 0 0 4 "${ui_options[@]}" 2>&1 1>$(tty)); code=$?; sudo setterm --cursor off
        [[ $code -eq 0 ]] && { sorting=$val; setPage; return; }
      ;;
    esac
  fi

  Redraw
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

Password() {
  local redraw=${1:-0}
  local stored_active=$active
  local key=$(grep -w -m1 'admin_menu_pass' "$HOMEDIR/ADMIN/.sys/.settings" | cut -d\' -f2 | base64 -d)
  local input=''

  clear; printf "\n ${C}PASS: ${NC}"; tput sc; for (( len=0; len<${#key}; len++ )); do printf "${C}_${NC}"; done; tput rc

  while [[ ${#input} -lt ${#key} ]]; do c=$(readInput); [[ ${#c} -gt 0 ]] && { case "$c" in 'ESC'|'S') break ;; *) input+=$(printf "$c" | sed 's/PGDOWN/X/;s/PGUP/Y/;s/ENTER/A/;s/SPACE/B/;s/UP/U/;s/DOWN/D/;s/LEFT/L/;s/RIGHT/R/'); tput rc; printf "${C}*${NC}"; tput sc ;; esac; }; done; clear; tput sc

  [[ $redraw -gt 0 ]] && Redraw
  [[ ${#input} -lt ${#key} ]] && return 2
  [[ "${input}" == "${key}" ]] && return 0

  return 1
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

checkCfg() {
  [ $# -lt 1 ] && return
  local game="${1}"
  local cfg_path="$HOMEDIR/CAFCA/cfg/.backup/mame-libretro"
  local cfg_bak=$(find "$cfg_path/mame2003/cfg" -maxdepth 1 -type f -iname "${game}.cfg.bak*" -printf '%TY-%Tm-%Td %TT %p\n' 2>/dev/null | sort | tail -n 1 | grep -o '[^ ]*$')
  local cfg_new="$roms_path/mame-libretro/mame2003/cfg/${game}.cfg"

  [ -f "$cfg_bak" ] && { diff "$cfg_new" "$cfg_bak" >/dev/null 2>&1; status=$?; } || { status=1; }
  [[ $status -gt 0 ]] && { sudo cp "$cfg_new" "$cfg_path/mame2003/cfg/${game}.cfg.bak$(date +%y%m%d%H%M%S)"; } #echo -e "VFD 7   CFG CHANGED" > /tmp/serial.fifo; }
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

printer() {
  [[ $# -lt 1 ]] && return; txt="${1}"; pad=''; str=''; mark="${2:-0}"; data="${3:-}"
  [[ $colors -gt 0 ]] && str="${C}";

  str+="${txt}"; strlen=${#str}

  if [[ $center -gt 0 ]]; then
    [[ $(( strlen % 2 )) > 0 ]] && strlen=$(( strlen - 1 )); strlen=$(( strlen / 2 ))
    length=$(( width / 2 )); length=$(( length - strlen ))
    printf -v pad '%*s' "$length"
  fi

  [[ $mark -eq 1 ]] && str="${MARK}${str}${UNMARK}"; str="${pad}${str}";
  [[ $colors -eq 1 ]] && str+="${NC}"

  if [[ ${#data} -gt 0 ]]; then
    tput sc; for i in $(seq 0 $(( width - (${#data} + 1) )) ); do printf ' '; done
    [[ $mark -eq 1 ]] && data="${MARK}${data}${UNMARK}"
    [[ $colors -eq 1 ]] && data="${C}$data${NC}"
    printf "${data}"; tput rc; echo -e "${str}"
  else
    echo -e "${str}"
  fi
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

readInput() {
  Event=''
  read -rs -N 1 -t 0.1 Event1 >/dev/null

  [[ $? -gt 127 ]] && return

  case "$Event1" in
    $'\163')          Event="START" 	;;
    $'\165')          Event="U" 	;;
    $'\144')          Event="D" 	;;
    $'\154')          Event="L" 	;;
    $'\162')          Event="R" 	;;
    $'\n')            Event="ENTER"   	;;
    ' ')              Event="SPACE"   	;;
    [[:blank:]])      Event="TAB"     	;;
    *)
      read -rsn5 -t 0.01 Event2 >/dev/null

      case "$Event2" in
        "[A")         Event="UP"      	;;
        "[B")         Event="DOWN"    	;;
        "[D")         Event="LEFT"    	;;
        "[C")         Event="RIGHT"   	;;
        "[5~")        Event="PGUP"    	;;
        "[6~")        Event="PGDOWN"  	;;
        "OS"|"[[D")   Event="F4"      	;;
        *)
          case "$Event1$Event2" in
            $'\E')    Event="ESC"     	;;
          esac
          ;;
      esac
  esac

  printf "${Event}"
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

initSpin() {
  for spin_msg in 'SPIN_SENS' 'SPIN_SPEED'; do sendSerial "$spin_msg 1"; done
  sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --add spinner
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

makeArr() { for i in $(seq 1 $#); do font_idx["${!i}"]="${uniq_idx[$i]}"; done; }

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

loadSettings() {
  while read line; do
    key=$(echo "$line" | cut -d= -f1)
    val=$(echo "$line" | cut -d= -f2- | sed "s/'//g")
    declare -g "${key}"="$val"; #echo "${key}: $val"
  done< <(sed -n '/# GLOBAL\|# GAMES/,/# EOF_GLOBAL\|# EOF_GAMES/ { /^page.*$/d; /path$/d; s/#.*$//;/^\s*$/d; s/\s*$//; p }' "$assets_path/.settings")
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

sendSerial() { [ $# -ge 1 ] && echo -e "$@" > $serial_path; }

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

setBlankTime() { [[ $# -gt 0 ]] && [[ $1 =~ ^[0-9]+$ ]] && echo -ne "\033[9;$1]" > "$console"; }

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

debug() { [[ $debug -eq 1 ]] && echo -e "$@"; }

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

quitMenu() {
  read -rs -t 0.1 void >/dev/null
  [[ $admin_mode -eq 0 ]] && toggleAdmin
  [[ $admin_mode -eq 1 ]] && exit || Redraw
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

exitFunc() {
  local signal=$1
  local JOBS=($(jobs -p)); [[ ${#JOBS[@]} -gt 0 ]] && sudo kill -9 "${JOBS[@]}" &> /dev/null

  sudo bash "$HOMEDIR/ADMIN/.sys/thd/states.sh" --remove joystick

  [[ $spinner_off -eq 0 ]] && for spin_msg in 'SPIN_OFF 0' 'SPIN_SENS 1' 'SPIN_SPEED 1'; do sendSerial "$spin_msg"; done
  [[ $hide_term -eq 1 ]] && { stty -F /dev/tty1 sane; sudo setterm --cursor on; }
  [[ $post_clear -eq 1 ]] && { clear; tput cup 0 0; tput sc; }

  [ ! -z $blanktime ] && setBlankTime "$((blanktime/60))"
  [ -f "$assets_path/fb0.dump" ] && { sudo rm "$assets_path/fb0.dump"; }

  sed -i "/UI=/ s/[=].*$/=\x27${prev_state}\x27/" "$HOMEDIR/ADMIN/.sys/.states"

  export PS1="$PS1_STORED"; exit $signal
}

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

Main
exit

# ________________________________________________________________________________________________________________________________________________________________________________________________________

