#include <Adafruit_GFX.h>

#include "GLCD.h"
#include <hardware/pwm.h>
#include <ArduinoJson.h>

#define MCU_CLOCK         (133000000.0)

#define LED_PIN           LED_BUILTIN
#define BGL_PIN           28
#define BGL_TIMEOUT       5000
#define BGL_FADEOUT       10

#define PWM_DEFAULT       128
#define PWM_MAX           255

#define EMPTY_LINE \
"                     "

#define MENU \
"{" \
  "\"menu\" : [" \
    "\"command\"" \
    "\"options\": [1,2,3,4]" \
  "]" \
"}"

// --------------------------------

JsonDocument menu;

constexpr uint8_t bgl_max = 255;
constexpr uint8_t row_N   = 4;
constexpr uint8_t col_N   = 6;
constexpr int16_t curSym  = 219;


enum : uint16_t {
  OFF  = 0,
  ON   = PWM_DEFAULT,
  FULL = PWM_MAX
};

enum : uint32_t {
  FIFO_IDLE,
  FIFO_READY,
  FIFO_BOOTED,
  FIFO_SEND
};

enum : uint8_t {
  UI_READ,
  UI_WRITE,
  UI_STDIN,
  UI_STDIN_DONE
};

enum : uint8_t {
  LCD_PRINT,
  LCD_PRINTLN,
  LCD_WRITE
};

// - - - - - - - - - - - -

uint8_t rowPins[row_N] = {
  14, // -> KP 2   ( R1 )
  11, // -> KP 5   ( R2 )
  9,  // -> KP 7   ( R3 )
  8   // -> KP 8   ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1   ( C1 )
  13, // -> KP 3   ( C2 )
  12, // -> KP 4   ( C3 )
  10, // -> KP 6   ( C4 )
  7,  // -> KP 9   ( C5 )
  6   // -> KP 10  ( C6 )
}; 

struct KeyMap {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  const char* t9_page;
};

KeyMap matrix[4][3] = {
  {
    { colPins[0], 1, 0, ".,!?:;_-+=*/'\"%$&#@(){}[]"},
    { colPins[2], 2, 0, "abcABC"},
    { colPins[1], 3, 0, "defDEF"},
  },
  {
    { colPins[0], 4, 0, "ghiGHI"},
    { colPins[2], 5, 0, "jklJKL"},
    { colPins[3], 6, 0, "mnoMNO"}
  },
  {
    { colPins[0], 7, 0, "pqrsPQRS"},
    { colPins[2], 8, 0, "tuvTUV"},
    { colPins[3], 9, 0, "wxyzWXYZ"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " "},
    { colPins[4], 12, 0, ""}
  },
};

// - - - - - - - - - - - -

struct PWM_struct {
  uint16_t      pin;
  bool          state;
  float         clk;
  uint16_t      pwm;
  uint16_t      pwm_prv;
  uint16_t      slice;
  unsigned long timer;
};

PWM_struct bgl_data = { 
  BGL_PIN, false, 519.5, ON, FULL, 0, 0
};


char           buf_input[256];
uint8_t        buf_index;
uint8_t        ui_mode;
uint16_t       bgl_fadeout;
unsigned long  bgl_timeout;
volatile bool  zero_booted;
unsigned long  kp_last_press, T;
bool           activeState = true;
String         buf;

// ---------------------------------------------

bool PWM_Init(PWM_struct &pwm_data = bgl_data);
bool GLCD_setBgl(uint16_t pwm, unsigned long t = BGL_TIMEOUT, bool state=false, PWM_struct &pwm_data = bgl_data);
bool checkInput();
void clearBuf();
void t9_reset();
KeyMap *readKeyPad();
// ---------------------------------------------

void setup()   {                
  Serial.begin(9600);
  
  pinMode( BGL_PIN, OUTPUT_12MA );
  digitalWrite( BGL_PIN, HIGH );
  
  // initialize KS0108 GLCD module with active high CS pins
  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false ) {
    Serial.println( F("display initialization failed!") );    // lack of RAM space
    while(true);  // stay here forever!
  }

  display.display();
  delay(2000); // Pause for 2 seconds
  display.clearDisplay();
  
  PWM_Init();

  for (int i = 0; i < row_N; i++)
    pinMode( rowPins[i],      OUTPUT ),
    digitalWrite( rowPins[i], HIGH   ),
    
    gpio_set_slew_rate( 
      rowPins[i], GPIO_SLEW_RATE_SLOW);
  
  for (int i = 0; i < col_N; i++)
    pinMode( colPins[i], INPUT_PULLUP );

  //while (!Serial.available())
  demo();
    
  buf = Serial.readStringUntil('\n');
  ui_mode = UI_READ;
  Serial.println("READY");

  display.clearDisplay();           // Clear the buffer
  display.setTextSize(1);           // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);  // Draw white text
  display.setCursor(0, 0);          // Start at top-left corner
  display.cp437(true);              // Use full 256 char 'Code Page 437' font
  display.setTextWrap(true);        // Toggle text wrap (else clip right).
  display.display();
  delay(1000);

  int16_t y = display.getCursorY();
  display.println("####### READY #######");
  display.display();
  delay(1000);
  
  //int16_t y = (display.getCursorY() > 0) ? display.getCursorY()-1 : 0;
  
  display.println("HERROOOOOOOOOOOOOOOOO");
  display.display();
  delay(2000);

  // test erase line
  display.setCursor(0, y); 
  display.setTextColor(KS0108_OFF);
  for (int i = 0; i < 21; i++) 
    display.write(curSym);
  display.display();
  delay(1000);
  display.setCursor(0, y);
  display.setTextColor(KS0108_ON);
  
  display.println("oi mate");
  display.display();
  delay(2000);
  
  
  display.setTextColor(KS0108_ON); 
  deserializeJson(menu, String(MENU));
  memset(buf_input, 0, sizeof(buf_input));
  kp_last_press = millis();
}
// ---------------------------------------------

// main loop (nothing here!)
void loop() {
  if (ui_mode == UI_READ) {
    KeyMap *obj = readKeyPad();
    
    if (obj != NULL) {
      uint8_t key = obj->key;
      //Serial.println(key);

      int16_t symb = 
        (key == 2) ? 24 :
        (key == 8) ? 25 :
        (key == 4) ? 27 : 
        (key == 6) ? 26 : 22;
  
      display.clearDisplay();
      display.setCursor(0,0);
      display.write(symb);
      display.display();
      
      delay(30);
    }
  }
  else if (ui_mode == UI_WRITE) {
    if (checkInput()) {
      display.clearDisplay();
      display.setCursor(0,0);
      display.print("> ");
      display.print(buf_input);
      display.write(curSym);
      display.write('\n');
      display.display();
      delay(30);
    }
  }
}

// ---------------------------------------------

KeyMap *readKeyPad() {
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      uint8_t col_pin = matrix[row][col].pin;
      uint8_t key = matrix[row][col].key;  
      
      if (digitalRead(col_pin) == LOW) {
        if (millis()-kp_last_press) {
          while (digitalRead(col_pin) == LOW); 
          digitalWrite(rowPins[row], HIGH);
          kp_last_press = millis();
          
          return &(matrix[row][col]);
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  return nullptr;
}
/* ------------------------------------------------- */

void t9_reset() {
  for (uint8_t x = 0; x < 4; x++) 
    for (uint8_t y = 0; y < 3; y++) 
      matrix[x][y].index = 0;
}
// ---------------------------------------------------------------

void clearBuf() {
  memset(buf_input, 0, sizeof(buf_input));
  buf_index = 0;
  t9_reset();
}
// ---------------------------------------------------------------

bool checkInput() {
  uint8_t key, index;
  char c;
  
  KeyMap *obj = readKeyPad();
  
  if (obj != NULL) {
    key = obj->key;

    if (obj->index < 1) {
      if (buf_input[buf_index] != '\0')
        buf_index = (key != 10 && key != 12) ? buf_index + 1 : buf_index;
      t9_reset();
    } 
    else {
      if (key > 10)  
        buf_index++;
    }
    
    if (key == 10) {
      buf_input[buf_index] = '\0';
      buf_index = (buf_index > 0) ? buf_index - 1 : 0;
    }
      
    index = obj->index % strlen(obj->t9_page);
    
    if (strlen(obj->t9_page) > 0) {
      buf_input[buf_index] = obj->t9_page[index];
      obj->index++;
    }
    return true;
  }
  return false;
}

// ---------------------------------------------

bool GLCD_setBgl(uint16_t pwm, unsigned long t, bool state, PWM_struct &pwm_data) {
  pwm_data.pwm     =  pwm;
  pwm_data.timer   =  ((state && !pwm_data.state) || (pwm == ON)) ? millis() : pwm_data.timer;
  pwm_data.state   =  ((pwm == ON) || (state)) ? true : false;
   
  pwm_set_gpio_level( pwm_data.pin,   pwm            );
  pwm_set_enabled(    pwm_data.slice, (bool)(pwm > 0));

  bgl_timeout = ((pwm == ON) && pwm_data.state) ? t : bgl_timeout;

  return pwm_data.state;
}
// ---------------------------------------------

bool PWM_Init(PWM_struct &pwm_data) {
  uint16_t pwm_max   =  255,
           pwm_pin   =  BGL_PIN,
           pwm_slice =  pwm_gpio_to_slice_num(pwm_pin);
           
  float    pwm_freq  =  1000.0,
           pwm_clk   =  (MCU_CLOCK/pwm_freq)/(float)(pwm_max+1);
           

  gpio_set_function(    pwm_pin,    GPIO_FUNC_PWM );

  pwm_set_clkdiv(       pwm_slice,  pwm_clk       );
  pwm_set_wrap(         pwm_slice,  pwm_max       );
  
  pwm_data.pin       =  pwm_pin;
  pwm_data.clk       =  pwm_clk;
  pwm_data.slice     =  pwm_slice;

  return GLCD_setBgl(ON);
}
// ---------------------------------------------
