#define MCU_CLOCK         (133000000.0)

#define LED_PIN           LED_BUILTIN
#define BGL_PIN           28
#define BGL_TIMEOUT       5000
#define BGL_FADEOUT       10

#define PWM_DEFAULT       128
#define PWM_MAX           255

#define KP_DEBOUNCE       30

#define NUMFLAKES         10
#define LOGO_HEIGHT       16
#define LOGO_WIDTH        16

#define ArrLen(x)         (sizeof(x)/sizeof(x[0]))

static const unsigned char PROGMEM logo_bmp[] =
{ 0b00000000, 0b11000000,
  0b00000001, 0b11000000,
  0b00000001, 0b11000000,
  0b00000011, 0b11100000,
  0b11110011, 0b11100000,
  0b11111110, 0b11111000,
  0b01111110, 0b11111111,
  0b00110011, 0b10011111,
  0b00011111, 0b11111100,
  0b00001101, 0b01110000,
  0b00011011, 0b10100000,
  0b00111111, 0b11100000,
  0b00111111, 0b11110000,
  0b01111100, 0b11110000,
  0b01110000, 0b01110000,
  0b00000000, 0b00110000 
};

// KS0108 GLCD library initialization according to the following connection:
// KS0108_GLCD(DI, RW, E, DB0, DB1, DB2, DB3, DB4, DB5, DB6, DB7, CS1, CS2, RES);
KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);  

// --------------------------------

constexpr uint8_t bgl_max = 255;
constexpr uint8_t row_N   = 4;
constexpr uint8_t col_N   = 6;
constexpr int16_t curSym  = 219;


enum : uint16_t {
  OFF  = 0,
  ON   = PWM_DEFAULT,
  FULL = PWM_MAX
};

enum : uint32_t {
  FIFO_IDLE,
  FIFO_READY,
  FIFO_BOOTED,
  FIFO_SEND
};

enum : uint8_t {
  UI_READ,
  UI_WRITE,
  UI_STDIN,
  UI_STDIN_DONE
};

enum : uint8_t {
  LCD_PRINT,
  LCD_PRINTLN,
  LCD_WRITE
};

// - - - - - - - - - - - -

uint8_t rowPins[row_N] = {
  14, // -> KP 2   ( R1 )
  11, // -> KP 5   ( R2 )
  9,  // -> KP 7   ( R3 )
  8   // -> KP 8   ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1   ( C1 )
  13, // -> KP 3   ( C2 )
  12, // -> KP 4   ( C3 )
  10, // -> KP 6   ( C4 )
  7,  // -> KP 9   ( C5 )
  6   // -> KP 10  ( C6 )
}; 

struct KeyMap {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  const char* t9_page;
};

KeyMap matrix[4][3] = {
  {
    { colPins[0], 1, 0, ".,!?:;_-+=*/'\"%$&#@(){}[]"},
    { colPins[2], 2, 0, "abcABC"},
    { colPins[1], 3, 0, "defDEF"},
  },
  {
    { colPins[0], 4, 0, "ghiGHI"},
    { colPins[2], 5, 0, "jklJKL"},
    { colPins[3], 6, 0, "mnoMNO"}
  },
  {
    { colPins[0], 7, 0, "pqrsPQRS"},
    { colPins[2], 8, 0, "tuvTUV"},
    { colPins[3], 9, 0, "wxyzWXYZ"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " "},
    { colPins[4], 12, 0, ""}
  },
};

// - - - - - - - - - - - -

struct PWM_struct {
  uint16_t      pin;
  bool          state;
  float         clk;
  uint16_t      pwm;
  uint16_t      pwm_prv;
  uint16_t      slice;
  unsigned long timer;
};

PWM_struct bgl_data = { 
  BGL_PIN, false, 519.5, ON, FULL, 0, 0
};

char           buf_input[256];
uint8_t        paths[8];
uint8_t        buf_index;
uint8_t        ui_mode;
uint16_t       bgl_fadeout;
unsigned long  bgl_timeout;
volatile bool  zero_booted = false;
unsigned long  kp_last_press, T;
bool           activeState = true;
String         buf;


uint8_t active_entry = 0;
uint8_t menu_level = 0;

const char *entries1[] = {
  "COMMANDS",
  "SETTINGS",
  "TOOLS",
  "SYSINFO"
};

// ---------------------------------------------

bool PWM_Init(PWM_struct &pwm_data = bgl_data);
bool GLCD_setBgl(uint16_t pwm, unsigned long t = BGL_TIMEOUT, bool state=false, PWM_struct &pwm_data = bgl_data);
void GLCD_getBgl(unsigned long t = bgl_timeout, PWM_struct &pwm_data = bgl_data);
void GLCD_clearHeader(int16_t y=0);
bool checkInput();
void clearBuf();
void t9_reset();
void menu(uint8_t active = active_entry);
void setBit(unsigned int &target, uint16_t x, bool set);
unsigned int rawKeyPad();
KeyMap *readKeyPad();

// ---------------------------------------------

void GLCD_clearHeader(int16_t y) {
  display.setCursor(0, y); 
  display.setTextColor( (display.isInverted()) ? KS0108_ON : KS0108_OFF);
  
  for (int i = 0; i < 21; i++) 
    display.write(curSym);
  display.display();

  display.setCursor(0, y);
  display.setTextColor( (display.isInverted()) ? KS0108_OFF : KS0108_ON); 
}
/* ------------------------------------------------- */

bool GLCD_setBgl(uint16_t pwm, unsigned long t, bool state, PWM_struct &pwm_data) {
  pwm_data.pwm     =  pwm;
  pwm_data.timer   =  ((state && !pwm_data.state) || (pwm == ON)) ? millis() : pwm_data.timer;
  pwm_data.state   =  ((pwm == ON) || (state)) ? true : false;
   
  pwm_set_gpio_level( pwm_data.pin,   pwm            );
  pwm_set_enabled(    pwm_data.slice, (bool)(pwm > 0));

  bgl_timeout = ((pwm == ON) && pwm_data.state) ? t : bgl_timeout;

  return pwm_data.state;
}
// ---------------------------------------------

void GLCD_getBgl(unsigned long t, PWM_struct &pwm_data) {
  if ((millis()-bgl_data.timer) > t) {
    if (bgl_data.state || bgl_data.pwm > 0) {
      if (bgl_fadeout > 0)
        GLCD_setBgl( --bgl_data.pwm ),
        delay(bgl_fadeout);
      else
        GLCD_setBgl(OFF);
    }
    else
      GLCD_setBgl(OFF);
  }
}

// ---------------------------------------------


bool PWM_Init(PWM_struct &pwm_data) {
  uint16_t pwm_max   =  FULL,
           pwm_pin   =  BGL_PIN,
           pwm_slice =  pwm_gpio_to_slice_num(pwm_pin);
           
  float    pwm_freq  =  1000.0,
           pwm_clk   =  (MCU_CLOCK/pwm_freq)/(float)(pwm_max+1);
           

  gpio_set_function(    pwm_pin,    GPIO_FUNC_PWM );

  pwm_set_clkdiv(       pwm_slice,  pwm_clk       );
  pwm_set_wrap(         pwm_slice,  pwm_max       );
  
  pwm_data.pin       =  pwm_pin;
  pwm_data.clk       =  pwm_clk;
  pwm_data.slice     =  pwm_slice;


  /*
  if (ON < PWM_MAX) {
    pwm_set_enabled( pwm_data.slice, true);

    for (int i = PWM_MAX; i--> ON; i-- )
      pwm_set_gpio_level( pwm_data.pin, i),
      delay(20);
  }
  */
  T = millis();

  return GLCD_setBgl(ON, 30000);
}

// ---------------------------------------------

void menu(uint8_t active) {
  GLCD_setBgl(ON);
  
  display.clearDisplay();
  display.setCursor(0,0);
  display.display();

  display.setCursor((8*6), 0);
  display.println((active==0) ? "MENU" : entries1[active]);
  
  display.drawLine(0, 8, display.width()-1, 8, KS0108_ON);
  display.println();
  
  for (int i=0; i<ArrLen(entries1); i++) {
    if (i == active)  
      display.setTextColor(KS0108_OFF, KS0108_ON);
    else
      display.setTextColor(KS0108_ON);
    display.println(entries1[i]);
    display.display();
    display.setTextColor(KS0108_ON);
  }
}

// _____________________________________________________
/*
 *  GLCD DEMOS
 */

void testlineerase(uint16_t del = 20) {
  int16_t i;
  uint16_t color = (display.isInverted()) ? KS0108_ON : KS0108_OFF;
  //display.clearDisplay(); // Clear display buffer
  
  for(i=0; i<display.height(); i+=2) {
    display.drawLine(0, i, display.width()-1, i, color);
    display.display();
    delay(del);
  }

  display.drawLine(0, i--, display.width()-1, i, color);
  display.display();
  delay(del);
  
  for(; i > 0; i-=2) {
    display.drawLine(0, i, display.width()-1, i, color);
    display.display();
    delay(del);
  }
}

void testdrawline() {
  int16_t i;

  display.clearDisplay(); // Clear display buffer

  for(i=0; i<display.width(); i+=4) {
    display.drawLine(0, 0, i, display.height()-1, KS0108_ON);
    display.display(); // Update screen with each newly-drawn line
    delay(1);
  }
  for(i=0; i<display.height(); i+=4) {
    display.drawLine(0, 0, display.width()-1, i, KS0108_ON);
    display.display();
    delay(1);
  }
  delay(250);

  display.clearDisplay();

  for(i=0; i<display.width(); i+=4) {
    display.drawLine(0, display.height()-1, i, 0, KS0108_ON);
    display.display();
    delay(1);
  }
  for(i=display.height()-1; i>=0; i-=4) {
    display.drawLine(0, display.height()-1, display.width()-1, i, KS0108_ON);
    display.display();
    delay(1);
  }
  delay(250);

  display.clearDisplay();

  for(i=display.width()-1; i>=0; i-=4) {
    display.drawLine(display.width()-1, display.height()-1, i, 0, KS0108_ON);
    display.display();
    delay(1);
  }
  for(i=display.height()-1; i>=0; i-=4) {
    display.drawLine(display.width()-1, display.height()-1, 0, i, KS0108_ON);
    display.display();
    delay(1);
  }
  delay(250);

  display.clearDisplay();

  for(i=0; i<display.height(); i+=4) {
    display.drawLine(display.width()-1, 0, 0, i, KS0108_ON);
    display.display();
    delay(1);
  }
  for(i=0; i<display.width(); i+=4) {
    display.drawLine(display.width()-1, 0, i, display.height()-1, KS0108_ON);
    display.display();
    delay(1);
  }

  delay(2000); // Pause for 2 seconds
}

void testdrawrect(void) {
  display.clearDisplay();

  for(int16_t i=0; i<display.height()/2; i+=2) {
    display.drawRect(i, i, display.width()-2*i, display.height()-2*i, KS0108_ON);
    display.display(); // Update screen with each newly-drawn rectangle
    delay(1);
  }

  delay(2000);
}

void testfillrect(void) {
  display.clearDisplay();

  for(int16_t i=0; i<display.height()/2; i+=3) {
    // The INVERSE color is used so rectangles alternate white/KS0108_ON
    display.fillRect(i, i, display.width()-i*2, display.height()-i*2, KS0108_INVERSE);
    display.display(); // Update screen with each newly-drawn rectangle
    delay(1);
  }

  delay(2000);
}

void testdrawcircle(void) {
  display.clearDisplay();

  for(int16_t i=0; i<max(display.width(),display.height())/2; i+=2) {
    display.drawCircle(display.width()/2, display.height()/2, i, KS0108_ON);
    display.display();
    delay(1);
  }

  delay(2000);
}

void testfillcircle(void) {
  display.clearDisplay();

  for(int16_t i=max(display.width(),display.height())/2; i>0; i-=3) {
    // The INVERSE color is used so circles alternate white/black
    display.fillCircle(display.width() / 2, display.height() / 2, i, KS0108_INVERSE);
    display.display(); // Update screen with each newly-drawn circle
    delay(1);
  }

  delay(2000);
}

void testdrawroundrect(void) {
  display.clearDisplay();

  for(int16_t i=0; i<display.height()/2-2; i+=2) {
    display.drawRoundRect(i, i, display.width()-2*i, display.height()-2*i,
      display.height()/4, KS0108_ON);
    display.display();
    delay(1);
  }

  delay(2000);
}

void testfillroundrect(void) {
  display.clearDisplay();

  for(int16_t i=0; i<display.height()/2-2; i+=2) {
    // The INVERSE color is used so round-rects alternate white/KS0108_ON
    display.fillRoundRect(i, i, display.width()-2*i, display.height()-2*i,
      display.height()/4, KS0108_INVERSE);
    display.display();
    delay(1);
  }

  delay(2000);
}

void testdrawtriangle(void) {
  display.clearDisplay();

  for(int16_t i=0; i<max(display.width(),display.height())/2; i+=5) {
    display.drawTriangle(
      display.width()/2  , display.height()/2-i,
      display.width()/2-i, display.height()/2+i,
      display.width()/2+i, display.height()/2+i, KS0108_ON);
    display.display();
    delay(1);
  }

  delay(2000);
}

void testfilltriangle(void) {
  display.clearDisplay();

  for(int16_t i=max(display.width(),display.height())/2; i>0; i-=5) {
    // The INVERSE color is used so triangles alternate white/black
    display.fillTriangle(
      display.width()/2  , display.height()/2-i,
      display.width()/2-i, display.height()/2+i,
      display.width()/2+i, display.height()/2+i, KS0108_INVERSE);
    display.display();
    delay(1);
  }

  delay(2000);
}

void testdrawchar(void) {
  display.clearDisplay();

  display.setTextSize(1);      // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON); // Draw white text
  display.setCursor(0, 0);     // Start at top-left corner
  display.cp437(true);         // Use full 256 char 'Code Page 437' font

  // Not all the characters will fit on the display. This is normal.
  // Library will draw what it can and the rest will be clipped.
  for(int16_t i=0; i<256; i++) {
    if(i == '\n') display.write(' ');
    else          display.write(i);
  }

  display.display();
  delay(2000);
}

void testdrawstyles(void) {
  display.clearDisplay();

  display.setTextSize(1);             // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);        // Draw white text
  display.setCursor(0,0);             // Start at top-left corner
  display.println(F("Hello, world!"));

  display.setTextColor(KS0108_OFF, KS0108_ON); // Draw 'inverse' text
  display.println(3.141592);

  display.setTextSize(2);             // Draw 2X-scale text
  display.setTextColor(KS0108_ON);
  display.print(F("0x")); display.println(0xDEADBEEF, HEX);

  display.display();
  delay(2000);
}

void testscrolltext(void) {
  display.clearDisplay();

  display.setTextSize(2); // Draw 2X-scale text
  display.setTextColor(KS0108_ON);
  display.setCursor(10, 0);
  display.println(F("scroll"));
  display.display();      // Show initial text
  delay(1000);

  // Scroll in various directions, pausing in-between:
  // scroll right
  for (uint8_t scroll = 0; scroll < 0x0F; scroll++) {
    display.scrollRight(1);
    display.display();
    delay(10);
  }
  delay(1000);
  // scroll left
  for (uint8_t scroll = 0; scroll < 0x0F; scroll++) {
    display.scrollLeft(1);
    display.display();
    delay(10);
  }
  delay(1000);
  // diagonal scroll right-up
  for (uint8_t scroll = 0; scroll < display.height()/2; scroll++) {
    display.scrollRight(1);
    display.scrollUp(1);
    display.display();
    delay(10);
  }
  delay(1000);
  // diagonal scroll left-up
  for (uint8_t scroll = 0; scroll < display.height()/2; scroll++) {
    display.scrollLeft(1);
    display.scrollUp(1);
    display.display();
    delay(10);
  }
  delay(1000);
}

void testdrawbitmap(void) {
  display.clearDisplay();

  display.drawBitmap(
    (display.width()  - LOGO_WIDTH ) / 2,
    (display.height() - LOGO_HEIGHT) / 2,
    logo_bmp, LOGO_WIDTH, LOGO_HEIGHT, 1);
  display.display();
  delay(1000);
}

#define XPOS   0 // Indexes into the 'icons' array in function below
#define YPOS   1
#define DELTAY 2

void testanimate(const uint8_t *bitmap, uint8_t w, uint8_t h) {
  int8_t i, f, icons[NUMFLAKES][3];

  // Initialize 'snowflake' positions
  for(f=0; f< NUMFLAKES; f++) {
    icons[f][XPOS]   = random(1 - LOGO_WIDTH, display.width());
    icons[f][YPOS]   = -LOGO_HEIGHT;
    icons[f][DELTAY] = random(1, 6);
    Serial.print(F("x: "));
    Serial.print(icons[f][XPOS], DEC);
    Serial.print(F(" y: "));
    Serial.print(icons[f][YPOS], DEC);
    Serial.print(F(" dy: "));
    Serial.println(icons[f][DELTAY], DEC);
  }

  for(i=0; i<10; i++) { 
    display.clearDisplay(); // Clear the display buffer
    if (Serial.available() > 0) break;
    // Draw each snowflake:
    for(f=0; f< NUMFLAKES; f++) {
      display.drawBitmap(icons[f][XPOS], icons[f][YPOS], bitmap, w, h, KS0108_ON);
    }

    display.display(); // Show the display buffer on the screen
    delay(200);        // Pause for 1/10 second
    
    // Then update coordinates of each flake...
    for(f=0; f< NUMFLAKES; f++) {
      icons[f][YPOS] += icons[f][DELTAY];
      // If snowflake is off the bottom of the screen...
      if (icons[f][YPOS] >= display.height()) {
        // Reinitialize to a random position, just off the top
        icons[f][XPOS]   = random(1 - LOGO_WIDTH, display.width());
        icons[f][YPOS]   = -LOGO_HEIGHT;
        icons[f][DELTAY] = random(1, 6);
      }
    }
  }
}

void GLCD_animate(int del=20, int ripl=2) {
  int16_t i;
  unsigned long t = millis();
  
  display.clearDisplay();
  display.display();

  for(i = 0; i < max(display.width(),display.height())/2; i += 2) {
    if (i >= ripl)
      display.drawCircle( display.width()/2, display.height()/2, i-ripl, KS0108_OFF );
        
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_ON );    
    display.display();

    while ((millis()-t)<del) GLCD_getBgl();
    t = millis();
  }
  
  for(; i < (max(display.width(),display.height())/2) + (10+ripl); i += 2) {
    display.drawCircle( display.width()/2, display.height()/2, i-ripl, KS0108_OFF );
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_ON );
    display.display();
    delay(del);
  }
  delay(200);
}

void demo() {         
  while (Serial.available() == 0) {
    testdrawline();      // Draw many lines
    if(Serial.available() > 0) break;
    testdrawrect();      // Draw rectangles (outlines)
    if(Serial.available() > 0) break;
    testfillrect();      // Draw rectangles (filled)
    if(Serial.available() > 0) break;
    testdrawcircle();    // Draw circles (outlines)
    if(Serial.available() > 0) break;
    testfillcircle();    // Draw circles (filled)
    if(Serial.available() > 0) break;
    testdrawroundrect(); // Draw rounded rectangles (outlines)
    if(Serial.available() > 0) break;
    testfillroundrect(); // Draw rounded rectangles (filled)
    if(Serial.available() > 0) break;
    testdrawtriangle();  // Draw triangles (outlines)
    if(Serial.available() > 0) break;
    testfilltriangle();  // Draw triangles (filled)
    if(Serial.available() > 0) break;
    testdrawchar();      // Draw characters of the default font
    if(Serial.available() > 0) break;
    testdrawstyles();    // Draw 'stylized' characters
    if(Serial.available() > 0) break;
    testscrolltext();    // Draw scrolling text
    if(Serial.available() > 0) break;
    display.invertDisp(true);
    display.display();
    delay(1000);
    display.invertDisp(false);
    display.display();
    delay(1000);
    if(Serial.available() > 0) break;
    testdrawbitmap();    // Draw a small bitmap image
    if(Serial.available() > 0) break;  
    testanimate(logo_bmp, LOGO_WIDTH, LOGO_HEIGHT); // Animate bitmaps
    if(Serial.available() > 0) break;
  }
}
