#pragma once

#include <Arduino.h>
#include <stdint.h>

#define MANUFACTURER_NAME "data.dogenigt.dk"
#define DEVICE_NAME       "LO-PI Phone v1.0.0"
#define DEVICE_VID        0x1986
#define DEVICE_PID        0x0042  

#define ArrLen(x)         (sizeof(x)/sizeof(x[0]))

#define PIXELS            168
#define BGL_PIN           26

#define KP_DEBOUNCE       50
#define KP_HOLD_TIME      1000

KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);
const uint8_t ln_cmd = 7;

const uint8_t row_N = 4;
const uint8_t col_N = 6;
uint8_t rowPins[row_N] = {
  14, // -> KP 2  ( R1 )
  11, // -> KP 5  ( R2 )
  9,  // -> KP 7  ( R3 )
  8   // -> KP 8  ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1  ( C1 )
  13, // -> KP 3  ( C2 )
  12, // -> KP 4  ( C3 )
  10, // -> KP 6  ( C4 )
  7,  // -> KP 9  ( C5 )
  6   // -> KP 10 ( C6 )
}; 

char          kp_last_char  = '\0';
uint8_t       kp_last_key   = 0;
uint8_t       kp_last_index = 0;
uint8_t       kp_cursor     = 0;
unsigned long kp_last_press;

struct Map {
  uint8_t     pin;
  uint8_t     key;
  uint8_t     t9_index;
  const char* t9_dict;
};

Map matrix[4][3] = {
  {
    { colPins[0], 1, 0, ".,:;-+/"},
    { colPins[2], 2, 0, "abcABC"},
    { colPins[1], 3, 0, "defDEF"}
  },
  {
    { colPins[0], 4, 0, "ghiGHI"},
    { colPins[2], 5, 0, "jklJKL"},
    { colPins[3], 6, 0, "mnoMNO"}
  },
  {
    { colPins[0], 7, 0, "pqrsPQRS"},
    { colPins[2], 8, 0, "tuvTUV"},
    { colPins[3], 9, 0, "wxyzWXYZ"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " "},  
    { colPins[4], 12, 0, ""}
  },
};


// 21 x 8 chars = 168 (+ null byte)
char fbuf[PIXELS+1] = {'\0'};

//-------------------------------------------------------

void drawCmd(void) {
  display.clearDisplay();
  
  display.setTextSize(1); 
  display.setTextColor(KS0108_ON);
  display.setCursor(0, 0);
  display.cp437(true);
  
  display.write('>');
  for (int i=0; i<=kp_cursor; i++) {
    if (fbuf[i] != '\0')
      display.write(fbuf[i]);
  }
  display.write(219);
  display.display();
}

//-------------------------------------------------------

void t9_reset() {
  for (uint8_t x = 0; x < 4; x++) {
    for (uint8_t y = 0; y < 3; y++) 
      matrix[x][y].t9_index = 0;
  }
}

//-------------------------------------------------------

void t9_getChar(Map *btn) {
  uint8_t len = strlen(btn->t9_dict);
  uint8_t key = btn->key;
  uint8_t index;
  char chr;
  
  if (key == 10 || key == 12) {
    if (key == 10) {
      if (strlen(fbuf)) {
        fbuf[kp_cursor] = '\0';
        kp_cursor = ((kp_cursor > 0) ? kp_cursor-1 : 0);
      }
    }
    else if (key == 12) {
      if (strlen(fbuf) > 0 && (strlen(fbuf) < sizeof(fbuf)-1))
        fbuf[++kp_cursor] = '\r';
        return;
    }
  } 
  else {
    if (key == kp_last_key && key != 11) {
      if (len > 0)
        ++btn->t9_index %= len;
    }
    else {
      if (strlen(fbuf) > 0 && (strlen(fbuf) < sizeof(fbuf)-1))
        kp_cursor++;
      t9_reset();
    }

    index = btn->t9_index;
    chr = btn->t9_dict[index];
    fbuf[kp_cursor] = chr;
  }

  if (key == 11 && kp_last_key != 11) {
    if (strlen(fbuf)) {
      fbuf[kp_cursor] = '\0';
      kp_cursor = ((kp_cursor > 0) ? kp_cursor-1 : 0);
    }
  }

  kp_last_key = key;
}


//-------------------------------------------------------

void sendCmd(const char* cmd) {
  //Serial.print("cmd: "), Serial.println(cmd);
  
  Keyboard.println(cmd); delay(100);
  Keyboard.releaseAll();
  
  memset(fbuf,'\0',sizeof(fbuf));
  kp_cursor = 0;
  kp_last_key = 0;
  t9_reset();
}

//-------------------------------------------------------

void keyHeld(Map *btn) {
  uint8_t key = btn->key;

  switch(key) {
    case 1:
      break;
    case 10:
      break;
    case 11:
      break;
    case 12:
      break;
    default:
      break;
  }
}
