//#include <Arduino.h> 
#include <stdio.h>
//#include <pico/stdlib.h>
#include <hardware/gpio.h>
#include <hardware/uart.h>
#include <Adafruit_GFX.h>   // include adafruit GFX library
#include "KS0108_GLCD.h"    // include KS0108 GLCD library
#include <Fonts/Picopixel.h>

#define ArrLen(x) (sizeof(x)/sizeof(x[0]))

//#define DEBUG
//#define USE_MODEM
//#define USE_DISPLAY

#define GLCD_RST      2
#define MODEM_RST     28

#define sim800L       Serial1
#define DEBOUNCE      50


bool online;
bool glcd_on;
bool modem_ready;
bool sms_ready;
bool call_ready;

uint8_t sms_mode = 0;

const uint8_t row_N = 4;
const uint8_t col_N = 6;
unsigned long kp_last_press;

KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);

uint8_t rowPins[row_N] = {
  14, // -> KP 2  ( R1 )
  11, // -> KP 5  ( R2 )
  9,  // -> KP 7  ( R3 )
  8   // -> KP 8  ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1  ( C1 )
  13, // -> KP 3  ( C2 )
  12, // -> KP 4  ( C3 )
  10, // -> KP 6  ( C4 )
  7,  // -> KP 9  ( C5 )
  6   // -> KP 10 ( C6 )
}; 

char    buf_input[256];
uint8_t buf_index;

enum : uint8_t {
  NAV_NONE,
  NAV_UP_LEFT,
  NAV_UP,
  NAV_UP_RIGHT,
  NAV_LEFT,
  NAV_MIDDLE,
  NAV_RIGHT,
  NAV_DOWN_LEFT,
  NAV_DOWN,
  NAV_DOWN_RIGHT
};

struct Map {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  const char* t9_page;
};

Map matrix[4][3] = {
  {
    { colPins[0], 1, 0, ".,!?:;_-+=*/'\"%$&#@(){}[]"},
    { colPins[2], 2, 0, "abcABC"},
    { colPins[1], 3, 0, "defDEF"},
  },
  {
    { colPins[0], 4, 0, "ghiGHI"},
    { colPins[2], 5, 0, "jklJKL"},
    { colPins[3], 6, 0, "mnoMNO"}
  },
  {
    { colPins[0], 7, 0, "pqrsPQRS"},
    { colPins[2], 8, 0, "tuvTUV"},
    { colPins[3], 9, 0, "wxyzWXYZ"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " "},
    { colPins[4], 12, 0, ""}
  },
};
// ---------------------------------------------------------------

bool initGLCD() {

  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false ) 
    return false;
  
  display.clearDisplay();           // Clear the buffer
  display.setTextSize(1);           // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);  // Draw white text
  display.setCursor(0, 0);          // Start at top-left corner
  display.cp437(true);              // Use full 256 char 'Code Page 437' font
  display.setTextWrap(true);        // Set whether text should wrap around (else clip right).
  //display.setFont(&Picopixel);
  display.display();
  delay(1000);
  
  return true;
}
// ---------------------------------------------------------------

namespace Modem {
  String buf;
  unsigned long T;
  
  void Begin(int txPin=0, int rxPin=1) {        
    sim800L.setTX(txPin);
    sim800L.setRX(rxPin);
    sim800L.begin(9600);

    pinMode(MODEM_RST, OUTPUT);  
    digitalWrite(MODEM_RST, LOW); 
    delay(1000);
    digitalWrite(MODEM_RST, HIGH);
    delay(1000);
    
    while (!sim800L)
      if ((millis()-T) > 5000)
          break;
  }

  bool awaitResponse(bool showOutput=true, unsigned long timeout=5000) {
    T = millis();

    if (showOutput)
      display.clearDisplay(),
      display.setCursor(0, 0),
      display.display();
      
    while (sim800L.available() && (millis()-T) < timeout) {
      buf = sim800L.readString();
      
      if (buf.startsWith("OK") || buf.startsWith("ERROR"))
        return true;
      else {
        buf.trim();
        if (buf.length()) 
          display.println(buf),
          display.display();
      }
    }
    return false;
  }

  bool Init() {
    bool ready = false;
    int retries = 0;
    
    digitalWrite(MODEM_RST, LOW); delay(100);
    digitalWrite(MODEM_RST, HIGH);
    delay(1000);

    while (!ready && ++retries < 5) {
      sim800L.println("AT"); T = millis();

      while (!sim800L.available() && (millis()-T) < 5000);
      while (sim800L.available() > 0) {
        buf = sim800L.readStringUntil('\n'); buf.trim();

        if (buf == "OK") 
          ready = true; 
      }
      sim800L.read();
      
    }
    return ready;
  }


  void configure() {
    //const char *cmds[] = {
    String cmds[] = {
      "AT+CMEE=1",
      "AT+CMGF=1",
      "AT+CPMS=\"SM\",\"SM\",\"SM\"",
      "AT+CNMI=1,1,0,0,0" // "AT+CNMI=1,2,0,0,0" to only forward to TE
    };
    
    for (int i=0; i<ArrLen(cmds); i++) {
      if (cmds[i].startsWith("AT+CNMI=")) {
        if (sms_ready) {
          sim800L.println(cmds[i]); T = millis();  
        }
      }
      else {
        sim800L.println(cmds[i]); T = millis();  
      }
      
      while (!sim800L.available() && (millis()-T) < 5000);
      if (sim800L.available())
        display.clearDisplay(),
        display.setCursor(0, 0);
        
      while (sim800L.available()) {
        buf = sim800L.readStringUntil('\n');

        if (display.getCursorY() >= 48)
          display.pushUp(8);
        
        display.println(buf);
        display.display();
        
        buf.trim();

        if (buf.length())
          checkAT(buf);
      }
      delay(2000);
    }
  }
};

// ---------------------------------------------------------------

void lcdPrompt(const char* txt, int del=0) {
  display.clearDisplay(); 
  display.setCursor(0, 0); 
  display.print("> ");
  display.display();
  delay(del*2);
  
  for (int i = 0; i<strlen(txt); i++)
    display.write((char)txt[i]),
    display.display(),
    delay(del);
}
// ---------------------------------------------------------------


void setup() {
  delay(1000);
  Serial.begin(9600);

  
  #ifdef DEBUG
    while (!Serial);
  #endif
  
  // - - - - - - - - - - - - - - - - - 

  Modem::Begin(0,1);

  // - - - - - - - - - - - - - - - - - 
  
  for (int i = 0; i < row_N; i++)
    pinMode(rowPins[i], OUTPUT),
    gpio_set_slew_rate( rowPins[i], GPIO_SLEW_RATE_SLOW ),
    digitalWrite(rowPins[i], HIGH);
  
  for (int i = 0; i < col_N; i++)
    pinMode(colPins[i], INPUT_PULLUP);

  // - - - - - - - - - - - - - - - - - 

  glcd_on = initGLCD();

  
  display.println("BOOTING..");
  display.display();
  delay(1000);
  
  // - - - - - - - - - - - - - - - - - 

  modem_ready = Modem::Init();

  if (glcd_on) {
    display.clearDisplay(); 
    display.setCursor(0, 0); 
    display.print("MODEM: ");
    display.println((modem_ready)?"ONLINE":"OFFLINE");
    display.display();
  }
  delay(2000);

  Modem::configure();
  
  // - - - - - - - - - - - - - - - - - 

  lcdPrompt("Enter command", 100);
  
  memset(buf_input, 0, sizeof(buf_input));
  kp_last_press = millis();

}
// ---------------------------------------------------------------

Map *readKeyPad() {
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      uint8_t col_pin = matrix[row][col].pin;
      uint8_t key = matrix[row][col].key;  
      
      if (digitalRead(col_pin) == LOW) {
        if (millis()-kp_last_press) {
          while (digitalRead(col_pin) == LOW); 
          digitalWrite(rowPins[row], HIGH);
          kp_last_press = millis();
          
          return &(matrix[row][col]);
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  return nullptr;
}
// ---------------------------------------------------------------

void t9_reset() {
  for (uint8_t x = 0; x < 4; x++) 
    for (uint8_t y = 0; y < 3; y++) 
      matrix[x][y].index = 0;
}
// ---------------------------------------------------------------

void checkInput() {
  int16_t curSym = 219;
  uint8_t key, index;
  char c;
  
  Map *obj = readKeyPad();
  
  if (obj != NULL) {
    key = obj->key;

    if (obj->index < 1) {
      if (buf_input[buf_index] != '\0')
        buf_index = (key != 10 && key != 12) ? buf_index + 1 : buf_index;
      t9_reset();
    } 
    else {
      if (key > 10)  
        buf_index++;
    }
    
    if (key == 10) {
      buf_input[buf_index] = '\0';
      buf_index = (buf_index > 0) ? buf_index - 1 : 0;
    }
      
    index = obj->index % strlen(obj->t9_page);
    
    if (strlen(obj->t9_page) > 0) {
      buf_input[buf_index] = obj->t9_page[index];
      obj->index++;
    }
    
    //lcdPrompt("Enter command:");

    display.clearDisplay(); 
    display.setCursor(0, 0); 
    display.print("> ");  
    display.print(buf_input);
    display.write(curSym);
    display.write('\n');
    display.display();
    delay(100);
  }
}
// ---------------------------------------------------------------

char *str_parts(char *str, bool last=1, const char* delim = " ") {
  return (last) ? ( strchr(str,delim[0]) ? strchr(str,delim[0])+1 : str ) : strtok(str, delim);
}
// ---------------------------------------------------------------

void evalCmd(char* cmd, char* val) {
  for (int i=0; i<strlen(cmd); i++)
    cmd[i] = toupper(cmd[i]);

  if (cmd == "AT") {
    if (val != NULL)
      sim800L.println(val);
  } 
  else if (cmd == "SIM") {
    
  }
}
// ---------------------------------------------------------------

void checkAT(String response) {
  if (response == "SMS Ready")
    sms_ready = true;
  else if (response == "Call Ready")
    call_ready = true;
  else {
    if (response.indexOf(":") > -1) {
      if ( response.substring(0, response.indexOf(":")) == "+CMTI" ) {
        if (response.indexOf(",") > -1) {
          sim800L.print("AT+CMGR=");
          sim800L.println( 
            response.substring( response.indexOf(",")+1, response.length() ) 
          );
        }
      }
    }
  } 
}
// ---------------------------------------------------------------

void loop()  {
  String inBuf;
  unsigned long bs_timer;
  char *delim, *cmd, *val = nullptr;
  
  if (BOOTSEL) {
    bs_timer = millis();
    
    while (BOOTSEL);

    if (strlen(buf_input) > 0) {
      cmd = buf_input;
      delim = strchr(buf_input,' ');
      
      if (delim) 
        *delim = '\0',
        cmd = buf_input,
        val = delim + 1;
      
      evalCmd(cmd, val);
      delay(3000);

      display.clearDisplay(); 
      display.setCursor(0, 0); 
      display.print("> ");  
      display.display();
    }
    memset(buf_input, 0, sizeof(buf_input));
    t9_reset();
    buf_index = 0;
  }
  else if (Serial.available()) {
    inBuf = Serial.readString(); inBuf.trim();

    // AT+CMGS="+4592906275"
    if (!sms_mode) {
      if (inBuf.startsWith("AT+CMGS=")) {
        sim800L.print(inBuf);
        sim800L.print("\r");
        sms_mode = 1;
      }
      else {
        sim800L.println(inBuf);
      }           
    }
    else if (sms_mode == 1) {
      sim800L.print(inBuf);
      sim800L.write(0x1A);
      sms_mode = 0;
    }
    
    Modem::T = millis();
    while (!sim800L.available() && (millis()-Modem::T) < 5000);
  }
  else if (sim800L.available()) {
    display.clearDisplay(); 
    display.setCursor(0, 0);
    display.display();

    while (sim800L.available()) {
      inBuf = sim800L.readStringUntil('\n');

      if (display.getCursorY() >= 48)
        display.pushUp(8);
      
      display.println(inBuf);
      display.display();
      
      inBuf.trim();

      if (inBuf.length())
        checkAT(inBuf);
      /*
      inBuf = sim800L.readStringUntil('\n');
      Serial.print("> '"); Serial.print(inBuf); Serial.println("'");

      if (display.getCursorY() >= 48)
        display.pushUp(8);

      
      display.println(inBuf);
      display.display();
      inBuf.trim();
      
      if (inBuf.length())
        checkAT(inBuf);
      */
      /*
      if (inBuf.indexOf(":") > -1) {
        if ( inBuf.substring(0, inBuf.indexOf(":")) == "+CMTI" ) {
          if (inBuf.indexOf(",") > -1) {
            sim800L.print("AT+CMGR=");
            sim800L.println( 
              inBuf.substring( inBuf.indexOf(",")+1, inBuf.length() ) 
            );
          }
        }
      }
      */
    }
    delay(2000);
  }
  
  /*
  else if (sim800L.available()) {
    display.clearDisplay(); 
    display.setCursor(0, 0);
    display.display();

    while (sim800L.available())
      display.write(sim800L.read()),
      display.display();
    delay(2000);
  }
  */
  
  /*
  else if (sim800L.available()) {
    display.clearDisplay(); 
    display.setCursor(0, 0);
    display.println("INCOMING!"); 
    delay(2000);
    display.clearDisplay(); 
    display.setCursor(0, 0);

    while (sim800L.available()) {
      Modem::buf = sim800L.readStringUntil('\n'); 
      Modem::buf.trim();
      
      if (Modem::buf.length()) {
        if (display.getCursorY() >= 7)
          display.clearDisplay(),
          display.setCursor(0, 0),
          delay(1000);
          
        display.println(Modem::buf),
        display.display(),
        delay(500);
      }
    }
  }
  */
  else {
    checkInput();
  }  
}
