#pragma once

#define modemPort         Serial1
#define modemResetPin     28
// ______________________________________________________________________

// SIMOperatorFormat
enum : uint8_t {
    /// Automatically select the mobile network operator, simplifying the process.
    SIM_OPERATOR_FORMAT_AUTO,

    /// Manually specify and select the desired mobile network operator.
    SIM_OPERATOR_FORMAT_MANUAL,

    /// Deregister from the current network, suspending network registration.
    SIM_OPERATOR_FORMAT_DEREGISTER,

    /// Set operator without network registration, useful in specific scenarios.
    SIM_OPERATOR_FORMAT_SET_ONLY,

    /// Combine manual and automatic operator selection for flexibility.
    SIM_OPERATOR_FORMAT_MANUAL_AUTO
};
// ----------------------------------------

// SIMOperatorMode
enum : uint8_t { 
    /// Standard GSM (2G) operating mode.
    SIM_OPERATOR_MODE_GSM,

    /// Compact GSM operating mode, optimized for resource-constrained environments.
    SIM_OPERATOR_MODE_GSM_COMPACT,

    /// UTRAN (3G) operating mode, suitable for 3G network connectivity.
    SIM_OPERATOR_MODE_UTRAN,

    /// GSM with EGPRS (2.5G) operating mode, offering enhanced data rates.
    SIM_OPERATOR_MODE_GSM_EGPRS,

    /// UTRAN HSDPA operating mode, enabling high-speed data access.
    SIM_OPERATOR_MODE_UTRAN_HSDPA,

    /// UTRAN HSUPA operating mode, focused on high-speed uplink data transmission.
    SIM_OPERATOR_MODE_UTRAN_HSUPA,

    /// UTRAN HSDPA and HSUPA combined mode for versatile 3G connectivity.
    SIM_OPERATOR_MODE_UTRAN_HSDPA_HSUPA,

    /// E-UTRAN (4G) operating mode, for advanced 4G LTE network connectivity.
    SIM_OPERATOR_MODE_E_UTRAN
};
// ----------------------------------------

// SIM900PhonebookType
enum : uint8_t {
    /// National phonebook for storing local contacts.
    SIM_PHONEBOOK_NATIONAL       = 145,

    /// International phonebook for storing global contacts.
    SIM_PHONEBOOK_INTERNATIONAL  = 129,

    /// An unspecified or unknown phonebook type.
    SIM_PHONEBOOK_UNKNOWN        = 0
};
// ----------------------------------------

// SIM900CardService
enum : uint8_t {
    /// Asynchronous card service for data communication.
    SIM_CARD_SERVICE_ASYNC,

    /// Synchronous card service for coordinated data exchange.
    SIM_CARD_SERVICE_SYNC,

    /// PAD (Packet Assembler/Disassembler) access service.
    SIM_CARD_SERVICE_PAD_ACCESS,

    /// Packet service for data transmission.
    SIM_CARD_SERVICE_PACKET,

    /// Voice service for voice calls.
    SIM_CARD_SERVICE_VOICE,

    /// Fax service for facsimile communication.
    SIM_CARD_SERVICE_FAX
};
// ----------------------------------------

typedef struct _SIMSignal {
    /// Received Signal Strength Indication (RSSI) in decibels (dBm).
    uint8_t rssi;

    /// Bit Error Rate (BER) as a unitless ratio.
    uint8_t bit_error_rate;
}
SIMSignal;
// ----------------------------------------
  
typedef struct _SIMOperator {
    /// The operating mode of the mobile network operator.
    uint8_t mode;

    /// The operator selection format.
    uint8_t format;

    /// The name of the mobile network operator.
    String name;
} SIMOperator;
// ----------------------------------------

typedef struct _SIMRTC {
    /// Date component: day.
    uint8_t day;
    
    /// Date component: month.
    uint8_t month;
    
    /// Date component: year.
    uint8_t year;

    /// Time component: hour.
    uint8_t hour;
    
    /// Time component: minute.
    uint8_t minute;
    
    /// Time component: second.
    uint8_t second;

    /// GMT (Greenwich Mean Time) offset in hours.
    int8_t gmt;
} SIMRTC;
// ----------------------------------------
  
typedef struct _SIMAPN {
    /// The Access Point Name (APN) for data connectivity.
    String apn;

    /// The username for APN authentication.
    String username;

    /// The password for APN authentication.
    String password;
} SIMAPN;
// ----------------------------------------

typedef struct _SIMHTTPHeader {
    /// The header field key.
    String key;

    /// The header field value.
    String value;
} SIMHTTPHeader;
// ----------------------------------------

typedef struct _SIMHTTPRequest {
    /// The HTTP method for the request (e.g., GET, POST).
    String method;

    /// The data to be included in the request (e.g., POST data).
    String data;

    /// The domain or server to which the request is sent.
    String domain;

    /// The resource or URL path to access on the server.
    String resource;

    /// The status of the HTTP request.
    uint8_t status;

    /// The port on which the server is listening (e.g., 80 for HTTP).
    uint16_t port;

    /// An array of HTTP headers associated with the request.
    SIMHTTPHeader *headers;

    /// The number of HTTP headers in the array.
    uint16_t header_count;
} SIMHTTPRequest;
// ----------------------------------------

typedef struct _SIMHTTPResponse {
    /// The HTTP status code of the response.
    uint16_t status;

    /// An array of HTTP headers included in the response.
    SIMHTTPHeader *headers;

    /// The number of HTTP headers in the array.
    uint16_t header_count;

    /// The data received in the HTTP response, such as HTML content or JSON data.
    String data;
} SIMHTTPResponse;
// ----------------------------------------

typedef struct _SIMCardAccount {
    /// The name associated with the card account.
    String name;
    
    /// The card's phone number.
    String number;

    /// The card's type (e.g., SIM card).
    uint8_t type;
    
    /// The card's speed or data rate.
    uint8_t speed;

    /// The type of phonebook (national, international) where the number is stored.
    uint8_t numberType;

    /// The type of card service (e.g., voice, data) associated with the card account.
    uint8_t service;
} SIMCardAccount;
// ----------------------------------------

typedef struct _SIMPhonebookCapacity {
    /// The type of phonebook memory (e.g., "SM" for SIM memory).
    String memoryType;

    /// The number of entries used in the phonebook memory.
    uint8_t used;

    /// The maximum number of entries that can be stored in the phonebook memory.
    uint8_t max;
} SIMPhonebookCapacity;

// ______________________________________________________________________

namespace Modem {

  bool hasAPN = false;
  unsigned long T;
  
  // ----------------------------------------

  void init(int txPin=0, int rxPin=1) {
    modemPort.setTX(txPin);
    modemPort.setRX(rxPin);
    modemPort.begin(9600);
  
    pinMode(modemResetPin, OUTPUT);
    digitalWrite(modemResetPin, HIGH);
    
    delay(5000);    
  }
  
  // ----------------------------------------

  void sendCommand(String message) {
      modemPort.print(message + "\r\n");
  }
  // ----------------------------------------

  String getResponse() {
    delay(500);
    T = millis();
    
    while ((millis()-T) < 5000 && modemPort.available() == 0);
    
    if (modemPort.available() > 0) {
        String response = modemPort.readString();
        response.trim();

        return response;
    }

    return "";
  }
  // ----------------------------------------

  String getReturnedMode() {
      String response = getResponse();
      return response.substring(response.lastIndexOf('\n') + 1);
  }  
  // ----------------------------------------

  bool isSuccessCommand() {
      return getReturnedMode() == F("OK");
  }
  // ----------------------------------------

  String rawQueryOnLine(uint16_t line) {
    String response = getResponse();
    String result = "";

    uint16_t currentLine = 0;
    for (int i = 0; i < response.length(); i++)
        if (currentLine == line && response[i] != '\n')
            result += response[i];
        else if (response[i] == '\n') {
            currentLine++;

            if (currentLine > line)
                break;
        }

    return result;
  }

  // ----------------------------------------
  
  String queryResult() {
      String response = getResponse();
      String result = F("");
  
      int idx = response.indexOf(": ");
      if (idx != -1)
          result = response.substring(
              idx + 2,
              response.indexOf('\n', idx)
          );
  
      return result;
  }
  // ----------------------------------------
  
  bool handshake() {
      sendCommand(F("AT"));
      return isSuccessCommand();
  }
  // ----------------------------------------
  
  bool isCardReady() {
      sendCommand(F("AT+CPIN?"));
      return isSuccessCommand();
  }
  // ----------------------------------------
  
  SIMSignal signal() {
      SIMSignal signal;
      signal.rssi = signal.bit_error_rate = 0;
      sendCommand("AT+CSQ");
  
      String response = queryResult();
      uint8_t delim = response.indexOf(',');
  
      if(delim == -1)
          return signal;
  
      signal.rssi = (uint8_t) response.substring(0, delim).toInt();
      signal.bit_error_rate = (uint8_t) response.substring(delim + 1).toInt();
  
      return signal;
  }
  // ----------------------------------------
  
  void close() {
      modemPort.end();
  }
  // ----------------------------------------

  bool hangUp() {
      sendCommand(F("ATH"));
      return isSuccessCommand();
  }
  // ----------------------------------------

  bool sendSMS(String number, String message) {
      handshake();
  
      sendCommand(F("AT+CMGF=1"));
      delay(500);
      sendCommand("AT+CMGS=\"" + number + "\"");
      delay(500);
      sendCommand(message);
      delay(500);
      modemPort.write(0x1a);
  
      return getReturnedMode().startsWith(">");
  }
  // ----------------------------------------

  String readSMS() {
      String resp, response = "";
      
      handshake();
      
      sendCommand(F("AT+CMGF=1"));
      if(!isSuccessCommand())
          return response;
      //sendCommand("AT+CNMI=1,2,0,0,0");
      sendCommand(F("AT+CMGF=1"));
      if(!isSuccessCommand())
          return response;

      sendCommand("AT+CPMS=\"SM\"");
      resp = getResponse();
      //if(!isSuccessCommand())
      //    return response;
          
      sendCommand("AT+CMGL=\"ALL\"");
      delay(1500);
      //response = getResponseLines();
      //delay(1500);
      //if(!resp.endsWith(F("CONNECT OK")))
      //if (resp.length())
      return response;
  }
  // ----------------------------------------

  SIMOperator networkOperator() {
      SIMOperator simOperator;
      simOperator.mode = 0;
      simOperator.format = 0;
      simOperator.name = "";
  
      sendCommand(F("AT+COPS?"));
  
      String response = queryResult();
      uint8_t delim1 = response.indexOf(','),
          delim2 = response.indexOf(',', delim1 + 1);
  
      simOperator.mode = (uint8_t) response.substring(0, delim1).toInt();
      simOperator.format = (uint8_t) response.substring(delim1 + 1, delim2).toInt();
      simOperator.name = response.substring(delim2 + 2, response.length() - 2);
  
      return simOperator;
  }
  // ----------------------------------------
  
  bool connectAPN(SIMAPN apn) {
      sendCommand(F("AT+CMGF=1"));
      if(!isSuccessCommand())
          return false;
  
      sendCommand(F("AT+CGATT=1"));
      if(!isSuccessCommand())
          return false;
      
      sendCommand(
          "AT+CSTT=\"" + apn.apn +
          "\",\"" + apn.username +
          "\",\"" + apn.password + "\""
      );
  
      return (hasAPN = isSuccessCommand());
  }
  // ----------------------------------------
  
  bool enableGPRS() {
      if(!hasAPN)
          return false;
  
      sendCommand(F("AT+CIICR"));
      delay(1000);
  
      return isSuccessCommand();
  }
  // ----------------------------------------
  
  SIMHTTPResponse request(SIMHTTPRequest request) {
      SIMHTTPResponse response;
      response.status = -1;
  
      if(!hasAPN)
          return response;
  
      sendCommand(
          "AT+CIPSTART=\"TCP\",\"" + request.domain +
          "\"," + String(request.port)
      );
      
      String resp = getResponse();
      resp.trim();
  
      delay(1500);
      if(!resp.endsWith(F("CONNECT OK")))
          return response;
  
      String requestStr = request.method + " " +
          request.resource + " HTTP/1.0\r\nHost: " +
          request.domain + "\r\n";
  
      for(int i = 0; i < request.header_count; i++)
          requestStr += request.headers[i].key + ": " +
              request.headers[i].value + "\r\n";
  
      if(request.data != "" || request.data != NULL)
          requestStr += request.data + "\r\n";
  
      requestStr += F("\r\n");
      sendCommand(requestStr);
  
      // TODO
      return response;
  }
  // ----------------------------------------

  bool updateRtc(SIMRTC config) {
      sendCommand(
          "AT+CCLK=\"" + String(config.year <= 9 ? "0" : "") + String(config.year) +
          "/" + String(config.month <= 9 ? "0" : "") + String(config.month) +
          "/" + String(config.day <= 9 ? "0" : "") + String(config.day) +
          "," + String(config.hour <= 9 ? "0" : "") + String(config.hour) +
          ":" + String(config.minute <= 9 ? "0" : "") + String(config.minute) +
          ":" + String(config.second <= 9 ? "0" : "") + String(config.second) +
          "+" + String(config.gmt <= 9 ? "0" : "") + String(config.gmt) + "\""
      );
  
      return isSuccessCommand();
  }
  // ----------------------------------------

  SIMRTC rtc() {
      SIMRTC rtc;
      rtc.year = rtc.month = rtc.day =
          rtc.hour = rtc.minute = rtc.second = 
          rtc.gmt = 0;
  
      sendCommand(F("AT+CMGF=1"));
      if(!isSuccessCommand())
          return rtc;
  
      sendCommand(F("AT+CENG=3"));
      if(!isSuccessCommand())
          return rtc;
      sendCommand(F("AT+CCLK?"));
      
      String time = queryResult();
      time = time.substring(1, time.length() - 2);
  
      uint8_t delim1 = time.indexOf('/'),
          delim2 = time.indexOf('/', delim1 + 1),
          delim3 = time.indexOf(',', delim2),
          delim4 = time.indexOf(':', delim3),
          delim5 = time.indexOf(':', delim4 + 1),
          delim6 = time.indexOf('+', delim5);
  
      rtc.year =  (uint8_t) time.substring(0, delim1).toInt();
      rtc.month = (uint8_t) time.substring(delim1 + 1, delim2).toInt();
      rtc.day = (uint8_t) time.substring(delim2 + 1, delim3).toInt();
      rtc.hour = (uint8_t) time.substring(delim3 + 1, delim4).toInt();
      rtc.minute = (uint8_t) time.substring(delim4 + 1, delim5).toInt();
      rtc.second = (uint8_t) time.substring(delim5 + 1, delim6).toInt();
      rtc.gmt = (uint8_t) time.substring(delim6 + 1).toInt();
  
      return rtc; 
  }
  // ----------------------------------------

  bool savePhonebook(uint8_t index, SIMCardAccount account) {
      sendCommand(
          "AT+CPBW=" + String(index) +
          ",\"" + account.number +
          "\"," + account.numberType +
          ",\"" + account.name + "\""
      );
      return isSuccessCommand();
  }
  // ----------------------------------------

  SIMCardAccount retrievePhonebook(uint8_t index) {
      sendCommand("AT+CPBR=" + String(index));
  
      SIMCardAccount accountInfo;
      accountInfo.numberType = 0;
  
      String response = queryResult();
      response = response.substring(response.indexOf(',') + 1);
  
      uint8_t delim1 = response.indexOf(','),
          delim2 = response.indexOf(',', delim1 + 1);
  
      accountInfo.number = response.substring(1, delim1 - 1);
      
      uint8_t type = (uint8_t) response.substring(delim1 + 1, delim2).toInt();
      if(type == 129 || type == 145)
          accountInfo.numberType = type;
      else accountInfo.numberType = 0;
  
      accountInfo.name = response.substring(delim2 + 2, response.length() - 2);
      return accountInfo;
  }
  // ----------------------------------------

  bool deletePhonebook(uint8_t index) {
      sendCommand("AT+CPBW=" + String(index));
      return isSuccessCommand();
  }
  // ----------------------------------------
  
  SIMPhonebookCapacity phonebookCapacity() {
      SIMPhonebookCapacity capacity;
      capacity.used = capacity.max = 0;
      capacity.memoryType = F("");
  
      sendCommand("AT+CPBS?");
  
      String response = queryResult();
      uint8_t delim1 = response.indexOf(','),
          delim2 = response.indexOf(',', delim1 + 1);
  
      capacity.memoryType = response.substring(1, delim1 - 1);
      capacity.used = (uint8_t) response.substring(delim1 + 1, delim2).toInt();
      capacity.max = (uint8_t) response.substring(delim2 + 1).toInt();
  
      return capacity;
  }
  // ----------------------------------------
  
  SIMCardAccount cardNumber() {
      sendCommand(F("AT+CNUM"));
  
      SIMCardAccount account;
      account.name = F("");
  
      String response = queryResult();
      if(response == F(""))
          return account;
  
      uint8_t delim1 = response.indexOf(','),
          delim2 = response.indexOf(',', delim1 + 1),
          delim3 = response.indexOf(',', delim2 + 1),
          delim4 = response.indexOf(',', delim3 + 1);
  
      account.name = response.substring(1, delim1 - 1);
      account.number = response.substring(delim1 + 2, delim2 - 1);
      account.type = (uint8_t) response.substring(delim2 + 1, delim3).toInt();
      account.speed = (uint8_t) response.substring(delim3 + 1, delim4).toInt();
      account.service = (uint8_t) response.substring(delim4 + 1).toInt();
      account.numberType = 0;
  
      return account;
  }
  // ----------------------------------------

  String manufacturer() {
      sendCommand(F("AT+GMI"));
      return rawQueryOnLine(2);
  }
  // ----------------------------------------
  
  String softwareRelease() {
      sendCommand(F("AT+GMR"));
  
      String result = rawQueryOnLine(2);
      result = result.substring(result.lastIndexOf(F(":")) + 1);
  
      return result;
  }
  // ----------------------------------------
  
  String imei() {
      sendCommand(F("AT+GSN"));
      return rawQueryOnLine(2);
  }
  // ----------------------------------------
  
  String chipModel() {
      sendCommand(F("AT+GMM"));
      return rawQueryOnLine(2);
  }
  // ----------------------------------------
  
  String chipName() {
      sendCommand(F("AT+GOI"));
      return rawQueryOnLine(2);
  }
  // ----------------------------------------
  
  String ipAddress() {
      sendCommand(F("AT+CIFSR"));
      return rawQueryOnLine(2);
  }
  // ----------------------------------------
  void comm() {
    while (Serial.available() != 0) 
      modemPort.write(Serial.read());
  
    while (modemPort.available() != 0) 
      Serial.write(modemPort.read());
  }
};
