// Demonstrates a simple use of the setup1()/loop1() functions
// for a multiprocessor run.

#include <stdio.h>
#include <hardware/uart.h>
#include <Adafruit_GFX.h>
#include "KS0108_GLCD.h"
//#include "pico/stdlib.h"
//#include "pico/multicore.h"

#define sim800L     Serial1
#define SIM800_RST  28
#define DEBOUNCE    50

KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);

/* __________________________________________________________ */

const uint8_t row_N = 4;
const uint8_t col_N = 6;
volatile bool core0_ready = false;
unsigned long kp_last_press;

String buff_in;
String buff_out;

char    buf_input[256];
uint8_t buf_index;
       
// ------------------------------------------

uint8_t rowPins[row_N] = {
  14, // -> KP 2  ( R1 )
  11, // -> KP 5  ( R2 )
  9,  // -> KP 7  ( R3 )
  8   // -> KP 8  ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1  ( C1 )
  13, // -> KP 3  ( C2 )
  12, // -> KP 4  ( C3 )
  10, // -> KP 6  ( C4 )
  7,  // -> KP 9  ( C5 )
  6   // -> KP 10 ( C6 )
}; 

enum : uint8_t {
  NAV_NONE,
  NAV_UP_LEFT,
  NAV_UP,
  NAV_UP_RIGHT,
  NAV_LEFT,
  NAV_MIDDLE,
  NAV_RIGHT,
  NAV_DOWN_LEFT,
  NAV_DOWN,
  NAV_DOWN_RIGHT
};

struct Map {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  const char* t9_page;
};

Map matrix[4][3] = {
  {
    { colPins[0], 1, 0, ".,!?:;_-+=*/'\"%$&#@(){}[]"},
    { colPins[2], 2, 0, "abcABC"},
    { colPins[1], 3, 0, "defDEF"},
  },
  {
    { colPins[0], 4, 0, "ghiGHI"},
    { colPins[2], 5, 0, "jklJKL"},
    { colPins[3], 6, 0, "mnoMNO"}
  },
  {
    { colPins[0], 7, 0, "pqrsPQRS"},
    { colPins[2], 8, 0, "tuvTUV"},
    { colPins[3], 9, 0, "wxyzWXYZ"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " "},
    { colPins[4], 12, 0, ""}
  },
};

// ------------------------------------------

void ModemInit(int txPin=0, int rxPin=1) {
  sim800L.setTX(txPin);
  sim800L.setRX(rxPin);
  sim800L.begin(9600);

  pinMode(SIM800_RST, OUTPUT);
  digitalWrite(SIM800_RST, LOW); delay(100);
  digitalWrite(SIM800_RST, HIGH);
  
  delay(5000);
  while (sim800L.available())
    sim800L.read();
  //sim800L.print("AT\r\n");
}
// ------------------------------------------

bool initGLCD() {  
  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false ) 
    return false;
  
  display.clearDisplay();           // Clear the buffer
  display.setTextSize(1);           // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);  // Draw white text
  display.setCursor(0, 0);          // Start at top-left corner
  display.cp437(true);              // Use full 256 char 'Code Page 437' font
  display.setTextWrap(true);        // Set whether text should wrap around (else clip right).
  //display.setFont(&Picopixel);
  display.display();
  delay(1000);
  
  return true;
}
// ------------------------------------------

void lcdPrint(String msg) {
  if (display.getCursorY() >= 7)
    display.clearDisplay(),
    display.setCursor(0, 0);
    
  display.println(msg);
  display.display();
  delay(100);
}

/* __________________________________________________________ */

// C O R E  0

void setup() {
  delay(1000);
  Serial.begin(9600);

  ModemInit(0,1);
  
  // - - - - - - - - - - - - - - - - - 

  if (initGLCD())
    display.println("READY."),
    display.display(),
    delay(1000);
  
  // - - - - - - - - - - - - - - - - - 

  core0_ready = true;
}
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

void loop() {
  if (sim800L.available() > 0) {
    while (sim800L.available()) {
      buff_in = sim800L.readString();
      buff_in.trim();
      
      lcdPrint(buff_in);
      
      if (buff_in.length()) {
        Serial.print("SIM800L: '"); Serial.print(buff_in); Serial.println("'");

        if (buff_in.startsWith("OK") || buff_in.startsWith("ERROR")) {
          Serial.println(F("___________________________\n"));
          break;
        } 
      }
    }
    sim800L.read();
  }

  if (Serial.available() > 0) {
    while (Serial.available())  {
      buff_out = Serial.readString();
      buff_out.trim();
      sim800L.println(buff_out);
    }
  }
}

/* __________________________________________________________ */

// C O R E  1

void setup1() {
  delay(1000);

  while (!core0_ready) yield();
  Serial.begin(9600);

  // - - - - - - - - - - - - - - - - - 
    
  for (int i = 0; i < row_N; i++)
    pinMode(rowPins[i], OUTPUT),
    gpio_set_slew_rate( rowPins[i], GPIO_SLEW_RATE_SLOW ),
    digitalWrite(rowPins[i], HIGH);
  
  for (int i = 0; i < col_N; i++)
    pinMode(colPins[i], INPUT_PULLUP);
    
  // - - - - - - - - - - - - - - - - - 
    
  memset(buf_input, 0, sizeof(buf_input));
  kp_last_press = millis();
  
}
// -----------------------------------------------------------

char commands[5][16];

void loop1() {
  if (!BOOTSEL) {
    checkInput();
  }
  else {
    if (strlen(buf_input) > 0) {
      int i = 0;

      display.clearDisplay();
      display.setCursor(0,0);
      display.print("> ");
      display.display();
     
      Serial.print("input: '"); Serial.print(buf_input); Serial.println("'");

      if (strchr(buf_input, ' ')) {
        char *token = strtok(buf_input, " ");
        do {
          if (i < (sizeof(commands)/sizeof(commands[0]))) {
             Serial.print(i), Serial.print(": "), Serial.println(token);
             strncpy(commands[i++], token, strlen(token));
          }
          token = strtok(nullptr, " ");
        } while (token);
      }
      else {
        strncpy(commands[i++], buf_input, strlen(buf_input));
      }

      for (int n=0; n<i; n++) {
        Serial.print("commands[");
        Serial.print(n);
        Serial.print("] = '");
        Serial.print(commands[n]);
        Serial.println("'");
      }
      
      memset(buf_input, 0, sizeof(buf_input));
    }
  }
}

/* __________________________________________________________ */

Map *readKeyPad() {
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      uint8_t col_pin = matrix[row][col].pin;
      uint8_t key = matrix[row][col].key;  
      
      if (digitalRead(col_pin) == LOW) {
        //if ((millis()-kp_last_press) > DEBOUNCE && key != 12) {
        if (millis()-kp_last_press) {
          while (digitalRead(col_pin) == LOW); 
          digitalWrite(rowPins[row], HIGH);
          kp_last_press = millis();
          
          return &(matrix[row][col]);
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  return nullptr;
}
// ---------------------------------------------------------------

void t9_reset() {
  for (uint8_t x = 0; x < 4; x++) 
    for (uint8_t y = 0; y < 3; y++) 
      matrix[x][y].index = 0;
}
// ---------------------------------------------------------------

void checkInput() {
  int16_t curSym = 219;
  uint8_t key, index;
  char c;
  
  Map *obj = readKeyPad();
  
  if (obj != NULL) {
    key = obj->key;

    if (obj->index < 1) {
      if (buf_input[buf_index] != '\0')
        buf_index = (key != 10 && key != 12) ? buf_index + 1 : buf_index;
      t9_reset();
    } 
    else {
      if (key > 10)  
        buf_index++;
    }
    
    if (key == 10) {
      buf_input[buf_index] = '\0';
      buf_index = (buf_index > 0) ? buf_index - 1 : 0;
    }
    //if (key == 12) buf_index++;
      
    index = obj->index % strlen(obj->t9_page);
    
    if (strlen(obj->t9_page) > 0) {
      buf_input[buf_index] = obj->t9_page[index];
      obj->index++;
    }
    Serial.println(buf_input);
    
    display.clearDisplay();
    display.setCursor(0,0);
    display.print("> ");
    display.print(buf_input);
    display.write(curSym);
    display.write('\n');
    display.display();
    delay(100);
    
  }
}
// ---------------------------------------------------------------
