
/**********************************************************************************
 * 
 * Interfacing Arduino with KS0108 Monochrome GLCD.
 * This example is for KS0108 GLCD modules with 128x64 Pixel resolution (two CS pins).
 * This is a free software with NO WARRANTY - Use it at your own risk!
 * https://simple-circuit.com/
 *
***********************************************************************************
 Written by Limor Fried/Ladyada for Adafruit Industries,
 with contributions from the open source community.
 BSD license, check license.txt for more information
 All text above, and the splash screen below must be
 included in any redistribution.
************************************************************************************
* Modified to work with KS0108 monochrome GLCD. More information including circuit
*   diagram on:
* https://simple-circuit.com/
* 
 **********************************************************************************/

#define KS0108_NO_SPLASH

#include <Adafruit_GFX.h>   // include adafruit GFX library
#include "KS0108_GLCD.h"    // include KS0108 GLCD library


// KS0108 GLCD library initialization according to the following connection:
// KS0108_GLCD(DI, RW, E, DB0, DB1, DB2, DB3, DB4, DB5, DB6, DB7, CS1, CS2, RES);
KS0108_GLCD display = KS0108_GLCD(22, 27, 20, 21, 19, 18, 17, 16, 0, 1, 2, 3, 4, 5);

#define BGL_PIN       6


#define NUMFLAKES     10 // Number of snowflakes in the animation example
#define LOGO_HEIGHT   16
#define LOGO_WIDTH    16
static const unsigned char PROGMEM logo_bmp[] =
{ 0b00000000, 0b11000000,
  0b00000001, 0b11000000,
  0b00000001, 0b11000000,
  0b00000011, 0b11100000,
  0b11110011, 0b11100000,
  0b11111110, 0b11111000,
  0b01111110, 0b11111111,
  0b00110011, 0b10011111,
  0b00011111, 0b11111100,
  0b00001101, 0b01110000,
  0b00011011, 0b10100000,
  0b00111111, 0b11100000,
  0b00111111, 0b11110000,
  0b01111100, 0b11110000,
  0b01110000, 0b01110000,
  0b00000000, 0b00110000 };

void setup()   {                
  Serial.begin(9600);
  display.setTextSize(1);  

  // initialize KS0108 GLCD module with active high CS pins
  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false ) {
    Serial.println( F("display initialization failed!") );    // lack of RAM space
    while(true);  // stay here forever!
  }

  pinMode(BGL_PIN, OUTPUT);

  display.display();
  delay(2000); // Pause for 2 seconds

  for (;;) {
    // Clear the buffer
    display.clearDisplay();
    
    testdrawline();      // Draw many lines
    delay(2000);
    testdrawrect();      // Draw rectangles (outlines)
    delay(2000);
    testfillrect();      // Draw rectangles (filled)
    delay(2000);
    testdrawcircle();    // Draw circles (outlines)
    delay(2000);
    testfillcircle();    // Draw circles (filled)
    delay(2000);
    testdrawroundrect(); // Draw rounded rectangles (outlines)
    delay(2000);
    testfillroundrect(); // Draw rounded rectangles (filled)
    delay(2000);
    testdrawtriangle();  // Draw triangles (outlines)
    delay(2000);
    testfilltriangle();  // Draw triangles (filled)
    delay(2000);
    testdrawchar();      // Draw characters of the default font
    delay(2000);
    testdrawstyles();    // Draw 'stylized' characters
    delay(2000);
    testscrolltext();    // Draw scrolling text
    delay(2000);
    display.invertDisp(true);
    display.display();
    delay(1000);
    display.invertDisp(false);
    display.display();
    delay(1000);
  
    testdrawbitmap();    // Draw a small bitmap image
    delay(2000);
    testanimate(logo_bmp, LOGO_WIDTH, LOGO_HEIGHT); // Animate bitmaps
    delay(2000);
  }
}



void testdrawline() {
  int16_t i;

  display.clearDisplay(); // Clear display buffer
  digitalWrite(BGL_PIN, HIGH);
  
  for(i=0; i<display.width(); i+=4) {
    display.drawLine(0, 0, i, display.height()-1, KS0108_ON);
    display.display(); // Update screen with each newly-drawn line
    delay(1);
  }
  for(i=0; i<display.height(); i+=4) {
    display.drawLine(0, 0, display.width()-1, i, KS0108_ON);
    display.display();
    delay(1);
  }
  delay(250);

  display.clearDisplay();

  for(i=0; i<display.width(); i+=4) {
    display.drawLine(0, display.height()-1, i, 0, KS0108_ON);
    display.display();
    delay(1);
  }
  for(i=display.height()-1; i>=0; i-=4) {
    display.drawLine(0, display.height()-1, display.width()-1, i, KS0108_ON);
    display.display();
    delay(1);
  }
  delay(250);

  display.clearDisplay();

  for(i=display.width()-1; i>=0; i-=4) {
    display.drawLine(display.width()-1, display.height()-1, i, 0, KS0108_ON);
    display.display();
    delay(1);
  }
  for(i=display.height()-1; i>=0; i-=4) {
    display.drawLine(display.width()-1, display.height()-1, 0, i, KS0108_ON);
    display.display();
    delay(1);
  }
  delay(250);

  display.clearDisplay();

  for(i=0; i<display.height(); i+=4) {
    display.drawLine(display.width()-1, 0, 0, i, KS0108_ON);
    display.display();
    delay(1);
  }
  for(i=0; i<display.width(); i+=4) {
    display.drawLine(display.width()-1, 0, i, display.height()-1, KS0108_ON);
    display.display();
    delay(1);
  }

  delay(2000); // Pause for 2 seconds
  digitalWrite(BGL_PIN, LOW);
}

void testdrawrect(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);

  for(int16_t i=0; i<display.height()/2; i+=2) {
    display.drawRect(i, i, display.width()-2*i, display.height()-2*i, KS0108_ON);
    display.display(); // Update screen with each newly-drawn rectangle
    delay(1);
  }

  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testfillrect(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);

  for(int16_t i=0; i<display.height()/2; i+=3) {
    // The INVERSE color is used so rectangles alternate white/KS0108_ON
    display.fillRect(i, i, display.width()-i*2, display.height()-i*2, KS0108_INVERSE);
    display.display(); // Update screen with each newly-drawn rectangle
    delay(1);
  }

  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testdrawcircle(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  for(int16_t i=0; i<max(display.width(),display.height())/2; i+=2) {
    display.drawCircle(display.width()/2, display.height()/2, i, KS0108_ON);
    display.display();
    delay(1);
  }

  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testfillcircle(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  for(int16_t i=max(display.width(),display.height())/2; i>0; i-=3) {
    // The INVERSE color is used so circles alternate white/black
    display.fillCircle(display.width() / 2, display.height() / 2, i, KS0108_INVERSE);
    display.display(); // Update screen with each newly-drawn circle
    delay(1);
  }

  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testdrawroundrect(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  for(int16_t i=0; i<display.height()/2-2; i+=2) {
    display.drawRoundRect(i, i, display.width()-2*i, display.height()-2*i,
      display.height()/4, KS0108_ON);
    display.display();
    delay(1);
  }

  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testfillroundrect(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  for(int16_t i=0; i<display.height()/2-2; i+=2) {
    // The INVERSE color is used so round-rects alternate white/KS0108_ON
    display.fillRoundRect(i, i, display.width()-2*i, display.height()-2*i,
      display.height()/4, KS0108_INVERSE);
    display.display();
    delay(1);
  }

  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testdrawtriangle(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  for(int16_t i=0; i<max(display.width(),display.height())/2; i+=5) {
    display.drawTriangle(
      display.width()/2  , display.height()/2-i,
      display.width()/2-i, display.height()/2+i,
      display.width()/2+i, display.height()/2+i, KS0108_ON);
    display.display();
    delay(1);
  }

  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testfilltriangle(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  for(int16_t i=max(display.width(),display.height())/2; i>0; i-=5) {
    // The INVERSE color is used so triangles alternate white/black
    display.fillTriangle(
      display.width()/2  , display.height()/2-i,
      display.width()/2-i, display.height()/2+i,
      display.width()/2+i, display.height()/2+i, KS0108_INVERSE);
    display.display();
    delay(1);
  }

  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testdrawchar(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  display.setTextSize(1);      // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON); // Draw white text
  display.setCursor(0, 0);     // Start at top-left corner
  display.cp437(true);         // Use full 256 char 'Code Page 437' font

  // Not all the characters will fit on the display. This is normal.
  // Library will draw what it can and the rest will be clipped.
  for(int16_t i=0; i<256; i++) {
    if(i == '\n') display.write(' ');
    else          display.write(i);
  }

  display.display();
  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testdrawstyles(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  display.setTextSize(1);             // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);        // Draw white text
  display.setCursor(0,0);             // Start at top-left corner
  display.println(F("Hello, world!"));

  display.setTextColor(KS0108_OFF, KS0108_ON); // Draw 'inverse' text
  display.println(3.141592);

  display.setTextSize(2);             // Draw 2X-scale text
  display.setTextColor(KS0108_ON);
  display.print(F("0x")); display.println(0xDEADBEEF, HEX);

  display.display();
  delay(2000);
  digitalWrite(BGL_PIN, LOW);
}

void testscrolltext(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  display.setTextSize(2); // Draw 2X-scale text
  display.setTextColor(KS0108_ON);
  display.setCursor(10, 0);
  display.println(F("scroll"));
  display.display();      // Show initial text
  delay(1000);

  // Scroll in various directions, pausing in-between:
  // scroll right
  for (uint8_t scroll = 0; scroll < 0x0F; scroll++) {
    display.scrollRight(1);
    display.display();
    delay(10);
  }
  delay(1000);
  // scroll left
  for (uint8_t scroll = 0; scroll < 0x0F; scroll++) {
    display.scrollLeft(1);
    display.display();
    delay(10);
  }
  delay(1000);
  // diagonal scroll right-up
  for (uint8_t scroll = 0; scroll < display.height()/2; scroll++) {
    display.scrollRight(1);
    display.scrollUp(1);
    display.display();
    delay(10);
  }
  delay(1000);
  // diagonal scroll left-up
  for (uint8_t scroll = 0; scroll < display.height()/2; scroll++) {
    display.scrollLeft(1);
    display.scrollUp(1);
    display.display();
    delay(10);
  }
  delay(1000);
  digitalWrite(BGL_PIN, LOW);
}

void testdrawbitmap(void) {
  display.clearDisplay();
  digitalWrite(BGL_PIN, HIGH);
  
  display.drawBitmap(
    (display.width()  - LOGO_WIDTH ) / 2,
    (display.height() - LOGO_HEIGHT) / 2,
    logo_bmp, LOGO_WIDTH, LOGO_HEIGHT, 1);
  display.display();
  
  delay(1000);
  digitalWrite(BGL_PIN, LOW);
}

#define XPOS   0 // Indexes into the 'icons' array in function below
#define YPOS   1
#define DELTAY 2

void testanimate(const uint8_t *bitmap, uint8_t w, uint8_t h) {
  int8_t f, icons[NUMFLAKES][3];

  digitalWrite(BGL_PIN, HIGH);

  // Initialize 'snowflake' positions
  for(f=0; f< NUMFLAKES; f++) {
    icons[f][XPOS]   = random(1 - LOGO_WIDTH, display.width());
    icons[f][YPOS]   = -LOGO_HEIGHT;
    icons[f][DELTAY] = random(1, 6);
    Serial.print(F("x: "));
    Serial.print(icons[f][XPOS], DEC);
    Serial.print(F(" y: "));
    Serial.print(icons[f][YPOS], DEC);
    Serial.print(F(" dy: "));
    Serial.println(icons[f][DELTAY], DEC);
  }

  for(int i=0; i<100; i++) { // Loop forever...
    display.clearDisplay(); // Clear the display buffer

    // Draw each snowflake:
    for(f=0; f< NUMFLAKES; f++) {
      display.drawBitmap(icons[f][XPOS], icons[f][YPOS], bitmap, w, h, KS0108_ON);
    }

    display.display(); // Show the display buffer on the screen
    delay(100);        // Pause for 1/10 second

    // Then update coordinates of each flake...
    for(f=0; f< NUMFLAKES; f++) {
      icons[f][YPOS] += icons[f][DELTAY];
      // If snowflake is off the bottom of the screen...
      if (icons[f][YPOS] >= display.height()) {
        // Reinitialize to a random position, just off the top
        icons[f][XPOS]   = random(1 - LOGO_WIDTH, display.width());
        icons[f][YPOS]   = -LOGO_HEIGHT;
        icons[f][DELTAY] = random(1, 6);
      }
    }
  }
  digitalWrite(BGL_PIN, LOW);
}

void loop() {}
