#pragma once

/*
    r      Open text file for reading.  The stream is positioned at the
           beginning of the file.

    r+     Open for reading and writing.  The stream is positioned at the
           beginning of the file.

    w      Truncate file to zero length or create text file for writing.
           The stream is positioned at the beginning of the file.

    w+     Open for reading and writing.  The file is created if it does
           not exist, otherwise it is truncated.  The stream is
           positioned at the beginning of the file.

    a      Open for appending (writing at end of file).  The file is
           created if it does not exist.  The stream is positioned at the
           end of the file.

    a+     Open for reading and appending (writing at end of file).  The
           file is created if it does not exist.  The initial file
           position for reading is at the beginning of the file, but
           output is always appended to the end of the file.          */

#ifndef LFS_H
  enum lfs_open_flags {
    // open flags
    LFS_O_RDONLY        = 1,        // Open a file as read only
    LFS_O_WRONLY        = 2,        // Open a file as write only
    LFS_O_RDWR          = 3,        // Open a file as read and write
    LFS_O_CREAT         = 0x0100,   // Create a file if it does not exist
    LFS_O_EXCL          = 0x0200,   // Fail if a file already exists
    LFS_O_TRUNC         = 0x0400,   // Truncate the existing file to zero size
    LFS_O_APPEND        = 0x0800,   // Move to end of file on every write

    // internally used flags
    LFS_F_DIRTY         = 0x010000, // File does not match storage
    LFS_F_WRITING       = 0x020000, // File has been written since last flush
    LFS_F_READING       = 0x040000, // File has been read since last flush
    LFS_F_ERRED         = 0x080000, // An error occurred during write
    LFS_F_INLINE        = 0x100000, // Currently inlined in directory entry
  };
#endif

#ifdef USE_FONTS
  #include <Fonts/Picopixel.h>
  #include <Fonts/Tiny3x3a2pt7b.h>
  #include <Fonts/Org_01.h>
#endif

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#define MANUFACTURER      "data_dogenigt"
#define DEVICE_NAME       "LoPi-Phone v1.0"
#define DEVICE_VID        0x0920
#define DEVICE_PID        0x1986

#define BGL_TIMEOUT       5000
#define READ_DEL          250
#define BGL_PIN           28
#define LED_PIN           LED_BUILTIN

// _______________________________________________________________________

constexpr int curSym    = 219;
constexpr uint8_t row_N = 4;
constexpr uint8_t col_N = 6;

KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);

uint8_t rowPins[row_N] = {
  14, // -> KP 2  ( R1 )
  11, // -> KP 5  ( R2 )
  9,  // -> KP 7  ( R3 )
  8   // -> KP 8  ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1  ( C1 )
  13, // -> KP 3  ( C2 )
  12, // -> KP 4  ( C3 )
  10, // -> KP 6  ( C4 )
  7,  // -> KP 9  ( C5 )
  6   // -> KP 10 ( C6 )
}; 

// - - - - - - - - - - - -

enum : boolean {
  OFF, ON
};

enum : uint32_t {
  FIFO_IDLE,
  FIFO_READY,
  FIFO_BOOTED,
  FIFO_SEND
};

enum : uint8_t {
  UI_READ,
  UI_WRITE,
  UI_STDIN,
  UI_STDIN_DONE
};

struct Map {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  const char* t9_page;
};

Map matrix[4][3] = {
  {
    { colPins[0], 1, 0, ".,!?:;_-+=*/'\"%$&#@(){}[]"},
    { colPins[2], 2, 0, "abcABC"},
    { colPins[1], 3, 0, "defDEF"},
  },
  {
    { colPins[0], 4, 0, "ghiGHI"},
    { colPins[2], 5, 0, "jklJKL"},
    { colPins[3], 6, 0, "mnoMNO"}
  },
  {
    { colPins[0], 7, 0, "pqrsPQRS"},
    { colPins[2], 8, 0, "tuvTUV"},
    { colPins[3], 9, 0, "wxyzWXYZ"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " "},
    { colPins[4], 12, 0, ""}
  },
};

// - - - - - - - - - - - -

unsigned long bgl_timer;
unsigned long kp_last_press, T;
volatile bool zero_booted;

uint8_t       buf_index;
uint8_t       ui_mode;
char          buf_input[256];
uint32_t      fifo_msg;
bool          bgl_state;
String        buf;

File          file;

// ___________________________________________________

void pwm_init(uint16_t pin=BGL_PIN)
{
  #define MCU_CLOCK       (133000000.0) // 133MHz
  #define MCU_PWM_MAX_VAL (255.0)
  #define MCU_PWM_FREQ    (3000.0)
  
  // Set GPIO function to PWM
  gpio_set_function(pin, GPIO_FUNC_PWM);
  
  // Find out which PWM slice is connected to the given pin
  uint16_t slice_num = pwm_gpio_to_slice_num(pin);
  uint16_t wrap_val = OUTPUT_MCU_PWM_MAX_VAL;
  
  // Calculate clock divider
  float clk_div = (MCU_CLOCK / (float)CONFIG_MCU_PWM_FREQ) / (float)(OUTPUT_MCU_PWM_MAX_VAL + 1);
  
  // Check if the calculated clock divider is within valid range
  if (clk_div > 256.0 || clk_div < 1.0)
  {
    Serial.printf("Cannot calculate PWM clock divider, setting default clock to 1kHz\n");
    clk_div = (MCU_CLOCK / 1000.0) / (float)(OUTPUT_MCU_PWM_MAX_VAL + 1);
  }
  
  // Log information about the PWM setup
  Serial.printf("Pin:%d, PWM slice:%d, PWM channel:%d\n", pin, pwm_gpio_to_slice_num(pin), pwm_gpio_to_channel(pin));
  Serial.printf("PWM clock divider:%f, Wrap value:%d\n", clk_div, wrap_val);
  
  // Set PWM clock divider and wrap value
  pwm_set_clkdiv(slice_num, clk_div);
  pwm_set_wrap(slice_num, wrap_val);
  
  // Set default output to 50% duty cycle
  pwm_set_gpio_level(pin, wrap_val / 2);
  
  // Enable PWM
  pwm_set_enabled(slice_num, true);
}
/* ------------------------------------------------- */

void GLCD_setBgl(bool state, int bgl_pin=BGL_PIN) {
  digitalWrite( bgl_pin, bgl_state = state );
  bgl_timer = ( state ) ? millis() : bgl_timer;
}
/* ------------------------------------------------- */

bool GLCD_getBgl(unsigned long t=BGL_TIMEOUT) {
  if ((millis()-bgl_timer) > t)
    GLCD_setBgl(OFF);
    
  return bgl_state;
}
/* ------------------------------------------------- */

bool GLCD_Init() {  
  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false ) 
    return false;

  //display.setFont(&Picopixel);    // Set font, default = NULL
  display.clearDisplay();           // Clear the buffer
  display.setTextSize(1);           // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);  // Draw white text
  display.setCursor(0, 0);          // Start at top-left corner
  display.cp437(true);              // Use full 256 char 'Code Page 437' font
  display.setTextWrap(true);        // Toggle text wrap (else clip right).
  display.display();
  delay(1000);

  display.println("BOOTING.");
  display.display();
  
  GLCD_setBgl(ON);
  
  return true;
}
/* ------------------------------------------------- */

void GLCD_animate(int del=20, int ripl=2) {
  int16_t i;

  display.clearDisplay();
  display.display();

  for(i = 0; i < max(display.width(),display.height())/2; i += 2) {
    if (i >= ripl)
      display.drawCircle( 
        display.width()/2,
        display.height()/2,
        i-ripl,
        KS0108_OFF
      );
        
    display.drawCircle(
      display.width()/2,
      display.height()/2,
      i,
      KS0108_ON
    );
    
    display.display();
    delay(del);
  }
  
  for(; i < (max(display.width(),display.height())/2) + (10+ripl); i += 2) {
    display.drawCircle(
      display.width()/2,
      display.height()/2,
      i-ripl,
      KS0108_OFF
    );
    
    display.drawCircle(
      display.width()/2,
      display.height()/2,
      i,
      KS0108_ON
    );
    
    display.display();
    delay(del);
  }

  delay(200);
}
/* ------------------------------------------------- */

Map *readKeyPad() {
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      uint8_t col_pin = matrix[row][col].pin;
      uint8_t key = matrix[row][col].key;  
      
      if (digitalRead(col_pin) == LOW) {
        if (millis()-kp_last_press) {
          while (digitalRead(col_pin) == LOW); 
          digitalWrite(rowPins[row], HIGH);
          kp_last_press = millis();
          
          return &(matrix[row][col]);
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  return nullptr;
}
/* ------------------------------------------------- */

void t9_reset() {
  for (uint8_t x = 0; x < 4; x++) 
    for (uint8_t y = 0; y < 3; y++) 
      matrix[x][y].index = 0;
}
// ---------------------------------------------------------------

void clearBuf() {
  memset(buf_input, 0, sizeof(buf_input));
  buf_index = 0;
  t9_reset();
}
// ---------------------------------------------------------------

bool checkInput() {
  uint8_t key, index;
  char c;
  
  Map *obj = readKeyPad();
  
  if (obj != NULL) {
    key = obj->key;

    if (obj->index < 1) {
      if (buf_input[buf_index] != '\0')
        buf_index = (key != 10 && key != 12) ? buf_index + 1 : buf_index;
      t9_reset();
    } 
    else {
      if (key > 10)  
        buf_index++;
    }
    
    if (key == 10) {
      buf_input[buf_index] = '\0';
      buf_index = (buf_index > 0) ? buf_index - 1 : 0;
    }
      
    index = obj->index % strlen(obj->t9_page);
    
    if (strlen(obj->t9_page) > 0) {
      buf_input[buf_index] = obj->t9_page[index];
      obj->index++;
    }
    return true;
  }
  return false;
}

// ---------------------------------------------------------------

void GLCD_overflow(int maxY = 56) {
  if (display.getCursorY() > maxY)
    display.pushUp(8),
    display.setCursor(0, maxY);
}
// ---------------------------------------------------------------

bool checkSerial() {
  return (bool)(Serial.available() > 0);
}
// ---------------------------------------------------------------

void checkHeader(String header) {
  delay(1);
}

// ---------------------------------------------------------------

void fsWrite(File& f, String line, const char* mode="w", bool nl=true) {
  if (f) {
    f = LittleFS.open(f.fullName(), "w");
    f.printf(
      ((nl) ? "%s\n" : "%s"),
      line.c_str()
    );
    f.close();
  }
}
// ---------------------------------------------------------------

int fsRead(File& f, int start_pos = 0) {
  if (f) {
    if (start_pos < f.size())
      f.seek(start_pos);
    
    while (f.available()) {
      //int x = f.read(); display.write(x);
      display.write(f.read());
      display.display();
    }
  }
  return f.position();
}
