#pragma once

#include <pico/stdlib.h>
#include <hardware/pwm.h>

#ifdef USE_FONTS
  #include <Fonts/Picopixel.h>
  #include <Fonts/Tiny3x3a2pt7b.h>
  #include <Fonts/Org_01.h>
#endif

/*
#include <Arduino.h>
#include <stdio.h>
#include <pico/stdlib.h>
#include <hardware/pwm.h>
#include <hardware/gpio.h>

#include <LittleFS.h>

#if defined(USE_TINYUSB)
  #include <Adafruit_TinyUSB.h>
#elif defined(CFG_TUSB_MCU)
  #error Incompatible USB stack. Use Adafruit TinyUSB.
#endif

#include <TinyUSB_Devices.h>
*/

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#define LED_PIN           LED_BUILTIN
#define MCU_CLOCK         (133000000.0)
#define PWM_DEFAULT       128
#define PWM_MAX           255
#define BGL_TIMEOUT       5000
#define BGL_FADEOUT       10
#define BGL_PIN           28
#define READ_DEL          250
#define DEVICE_VID        0x0920
#define DEVICE_PID        0x1986

#define MANUFACTURER      "data_dogenigt"
#define DEVICE_NAME       "LoPi-Phone v1.0"

//#define LCD_PRINT(x) do { lcd.print (x);  } while (0)

// _______________________________________________________________________

constexpr uint8_t row_N   = 4;
constexpr uint8_t col_N   = 6;
constexpr int16_t curSym  = 219;
constexpr uint8_t bgl_max = 255;

Adafruit_USBD_MSC usb_msc;

KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);

uint8_t rowPins[row_N] = {
  14, // -> KP 2   ( R1 )
  11, // -> KP 5   ( R2 )
  9,  // -> KP 7   ( R3 )
  8   // -> KP 8   ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1   ( C1 )
  13, // -> KP 3   ( C2 )
  12, // -> KP 4   ( C3 )
  10, // -> KP 6   ( C4 )
  7,  // -> KP 9   ( C5 )
  6   // -> KP 10  ( C6 )
}; 

// - - - - - - - - - - - -

enum : uint16_t {
  OFF  = 0,
  ON   = PWM_DEFAULT,
  FULL = PWM_MAX
};

enum : uint32_t {
  FIFO_IDLE,
  FIFO_READY,
  FIFO_BOOTED,
  FIFO_SEND
};

enum : uint8_t {
  UI_READ,
  UI_WRITE,
  UI_STDIN,
  UI_STDIN_DONE
};

enum : uint8_t {
  LCD_PRINT,
  LCD_PRINTLN,
  LCD_WRITE
};

// - - - - - - - - - - - -

struct Map {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  const char* t9_page;
};

Map matrix[4][3] = {
  {
    { colPins[0], 1, 0, ".,!?:;_-+=*/'\"%$&#@(){}[]"},
    { colPins[2], 2, 0, "abcABC"},
    { colPins[1], 3, 0, "defDEF"},
  },
  {
    { colPins[0], 4, 0, "ghiGHI"},
    { colPins[2], 5, 0, "jklJKL"},
    { colPins[3], 6, 0, "mnoMNO"}
  },
  {
    { colPins[0], 7, 0, "pqrsPQRS"},
    { colPins[2], 8, 0, "tuvTUV"},
    { colPins[3], 9, 0, "wxyzWXYZ"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " "},
    { colPins[4], 12, 0, ""}
  },
};

// - - - - - - - - - - - -

struct PWM_struct {
  uint16_t      pin;
  bool          state;
  float         clk;
  uint16_t      pwm;
  uint16_t      pwm_prv;
  uint16_t      slice;
  unsigned long timer;
};

PWM_struct bgl_data = { 
  BGL_PIN, 
  false,
  519.5, 
  ON, 
  FULL, 
  0, 
  0
};

/* ------------------------------------------------- */

template <typename T> 
void GLCD_print(
  T txt, 
  uint8_t mode      = LCD_PRINT, 
  bool preClear     = false,
  KS0108_GLCD& disp = display
);
//void GLCD_print(char txt, uint8_t mode, bool preClear, KS0108_GLCD& disp);
void GLCD_print(
  char txt, 
  uint8_t mode      = LCD_PRINT, 
  bool preClear     = false,
  KS0108_GLCD& disp = display
){ disp.write(txt); }

// - - - - - - - - - - - -

char           buf_input[256];
uint8_t        buf_index;
uint8_t        ui_mode;
uint16_t       bgl_fadeout;
unsigned long  bgl_timeout;
unsigned long  kp_last_press, T;
volatile bool  zero_booted;
String         buf;
bool           activeState = true;
//File           file;

// ___________________________________________________

// Callback invoked when received READ10 command.
// Copy disk's data to buffer (up to bufsize) and
// return number of copied bytes (must be multiple of block size)
int32_t RamDisk_reading(uint32_t lba, void* buffer, uint32_t bufsize) {
  uint8_t const* addr = msc_disk[lba];
  memcpy(buffer, addr, bufsize);
  aa++;
  return bufsize;
}

// Callback invoked when received WRITE10 command.
// Process data in buffer to disk's storage and
// return number of written bytes (must be multiple of block size)
int32_t RamDisk_writing(uint32_t lba, uint8_t* buffer, uint32_t bufsize) {
  uint8_t* addr = msc_disk[lba];
  memcpy(addr, buffer, bufsize);
  bb++;
  return bufsize;
}

// Callback invoked when WRITE10 command is completed (status received and accepted by host).
// used to flush any pending cache.
void RamDisk_flushing(void) {
  cc++;
}

// Invoked when received Test Unit Ready command.
// return true allowing host to read/write this LUN e.g SD card inserted
bool RamDisk_ready(void) {
  unsigned long t = millis();
  bool state = false;
  
  while ((millis()-t)<1000) {
    if (BOOTSEL) {
      dd++;
      state = true;
      break;
    }
  }
  return true; //state;
}

void RamDiskBegin() {
  // Set disk vendor id, product id and revision with string up to 8, 16, 4 characters respectively
  usb_msc.setID("Adafruit", "Mass Storage", "1.0");

  // Set disk size
  usb_msc.setCapacity(DISK_BLOCK_NUM, DISK_BLOCK_SIZE);

  // Set callback
  usb_msc.setReadWriteCallback( RamDisk_reading, RamDisk_writing, RamDisk_flushing );

  // Set Lun ready (RAM disk is always ready)
  usb_msc.setUnitReady(true);

  //if BTN_EJECT
  usb_msc.setReadyCallback( RamDisk_ready );

  usb_msc.begin();  
}

// ___________________________________________________

bool GLCD_setBgl(uint16_t pwm, unsigned long t = BGL_TIMEOUT, bool state=false, PWM_struct &pwm_data = bgl_data) {
  pwm_data.pwm     =  pwm;
  pwm_data.timer   =  ((state && !pwm_data.state) || (pwm == ON)) ? millis() : pwm_data.timer;
  pwm_data.state   =  ((pwm == ON) || (state)) ? true : false;
   
  pwm_set_gpio_level( pwm_data.pin,   pwm            );
  pwm_set_enabled(    pwm_data.slice, (bool)(pwm > 0));

  bgl_timeout = ((pwm == ON) && pwm_data.state) ? t : bgl_timeout;

  return pwm_data.state;
}
// ------------------------------------------------- 

void GLCD_getBgl(unsigned long t = bgl_timeout, PWM_struct &pwm_data = bgl_data) {
  if ((millis()-bgl_data.timer) > t) {
    if (bgl_data.state || bgl_data.pwm > 0) {
      if (bgl_fadeout > 0)
        GLCD_setBgl( --bgl_data.pwm ),
        delay(bgl_fadeout);
      else
        GLCD_setBgl(OFF);
    }
    else
      GLCD_setBgl(OFF);
  }
}
/* ------------------------------------------------- */

bool PWM_Init(PWM_struct &pwm_data = bgl_data) {
  uint16_t pwm_max   =  255,
           pwm_pin   =  BGL_PIN,
           pwm_slice =  pwm_gpio_to_slice_num(pwm_pin);
           
  float    pwm_freq  =  1000.0,
           pwm_clk   =  (MCU_CLOCK/pwm_freq)/(float)(pwm_max+1);
           

  gpio_set_function(    pwm_pin,    GPIO_FUNC_PWM );

  pwm_set_clkdiv(       pwm_slice,  pwm_clk       );
  pwm_set_wrap(         pwm_slice,  pwm_max       );
  
  pwm_data.pin       =  pwm_pin;
  pwm_data.clk       =  pwm_clk;
  pwm_data.slice     =  pwm_slice;

  return GLCD_setBgl(ON);
}
/* ------------------------------------------------- */

void GLCD_overflow(int maxY = 56) {
  if (display.getCursorY() > maxY)
    display.pushUp(8),
    display.setCursor(0, maxY);
}
// ---------------------------------------------------------------

void GLCD_animate(int del=20, int ripl=2) {
  int16_t i;
  unsigned long t = millis();
  
  display.clearDisplay();
  display.display();

  for(i = 0; i < max(display.width(),display.height())/2; i += 2) {
    if (i >= ripl)
      display.drawCircle( display.width()/2, display.height()/2, i-ripl, KS0108_OFF );
        
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_ON );    
    display.display();

    while ((millis()-t)<del) GLCD_getBgl();
    t = millis();
  }
  
  for(; i < (max(display.width(),display.height())/2) + (10+ripl); i += 2) {
    display.drawCircle( display.width()/2, display.height()/2, i-ripl, KS0108_OFF );
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_ON );
    display.display();
    delay(del);
  }
  delay(200);
}
/* ------------------------------------------------- */

bool GLCD_Init() {  
  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false ) 
    return false;

  //display.setFont(&Picopixel);    // Set font, default = NULL
  display.clearDisplay();           // Clear the buffer
  display.setTextSize(1);           // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);  // Draw white text
  display.setCursor(0, 0);          // Start at top-left corner
  display.cp437(true);              // Use full 256 char 'Code Page 437' font
  display.setTextWrap(true);        // Toggle text wrap (else clip right).
  display.display();
  delay(1000);
  
  bgl_fadeout = (BGL_FADEOUT) ? BGL_FADEOUT : 0;
  bgl_timeout = BGL_TIMEOUT;
  
  GLCD_setBgl(ON);
  
  return true;
}
/* ------------------------------------------------- */ 


template <typename T> void GLCD_print(T txt, uint8_t mode, bool preClear, KS0108_GLCD& disp) 
{
  GLCD_setBgl(ON);
  
  if (preClear)
    display.clearDisplay(),
    display.setCursor(0,0);
    
  switch(mode) {
    default:
    case 0:
      disp.print(txt);
      break;
    case 1:
      disp.println(txt);
      break;
    case 2:
      disp.print(txt);
      break;
  }
  display.display();
}

/* ------------------------------------------------- */

Map *readKeyPad() {
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      uint8_t col_pin = matrix[row][col].pin;
      uint8_t key = matrix[row][col].key;  
      
      if (digitalRead(col_pin) == LOW) {
        if (millis()-kp_last_press) {
          while (digitalRead(col_pin) == LOW); 
          digitalWrite(rowPins[row], HIGH);
          kp_last_press = millis();
          
          return &(matrix[row][col]);
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  return nullptr;
}
/* ------------------------------------------------- */

void t9_reset() {
  for (uint8_t x = 0; x < 4; x++) 
    for (uint8_t y = 0; y < 3; y++) 
      matrix[x][y].index = 0;
}
// ---------------------------------------------------------------

void clearBuf() {
  memset(buf_input, 0, sizeof(buf_input));
  buf_index = 0;
  t9_reset();
}
// ---------------------------------------------------------------

bool checkInput() {
  uint8_t key, index;
  char c;
  
  Map *obj = readKeyPad();
  
  if (obj != NULL) {
    key = obj->key;

    if (obj->index < 1) {
      if (buf_input[buf_index] != '\0')
        buf_index = (key != 10 && key != 12) ? buf_index + 1 : buf_index;
      t9_reset();
    } 
    else {
      if (key > 10)  
        buf_index++;
    }
    
    if (key == 10) {
      buf_input[buf_index] = '\0';
      buf_index = (buf_index > 0) ? buf_index - 1 : 0;
    }
      
    index = obj->index % strlen(obj->t9_page);
    
    if (strlen(obj->t9_page) > 0) {
      buf_input[buf_index] = obj->t9_page[index];
      obj->index++;
    }
    return true;
  }
  return false;
}

// ---------------------------------------------------------------

//bool checkSerial() { return (bool)(Serial.available() > 0); }
bool checkSerial(bool discard=false) {
  if (!Serial.available()) 
      return false;
  else if (discard) 
      Serial.readStringUntil('\n');
  return true;
}

// ---------------------------------------------------------------

void checkHeader(String header) {
  delay(1);
}

// ---------------------------------------------------------------
/*
void fsWrite(File& f, String line, const char* mode="w", bool nl=true) {
  if (f) {
    f = LittleFS.open(f.fullName(), "w");
    f.printf(
      ((nl) ? "%s\n" : "%s"),
      line.c_str()
    );
    f.close();
  }
}
// ---------------------------------------------------------------

int fsRead(File& f, int start_pos = 0) {
  if (f) {
    if (start_pos < f.size())
      f.seek(start_pos);
    
    while (f.available()) {
      //int x = f.read(); display.write(x);
      display.write(f.read());
      display.display();
    }
  }
  return f.position();
}
*/
