#pragma once

#include <pico/stdlib.h>
#include <hardware/pwm.h>

#ifdef USE_FONTS
  #include <Fonts/Picopixel.h>
  #include <Fonts/Tiny3x3a2pt7b.h>
  #include <Fonts/Org_01.h>
#endif

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#define ArrLen(x)         (sizeof(x)/sizeof(x[0]))

#define LED_PIN           LED_BUILTIN
#define MCU_CLOCK         (133000000.0)
#define PWM_DEFAULT       255
#define PWM_MAX           255
#define BGL_TIMEOUT       5000
#define BGL_FADEOUT       10
#define BGL_PIN           28
#define READ_DEL          250
#define KP_DEBOUNCE       30

#define DEVICE_VID        0x0920
#define DEVICE_PID        0x1986
#define MANUFACTURER      "data_dogenigt"
#define DEVICE_NAME       "LoPi-Phone v1.0"

//#define LCD_PRINT(x) do { lcd.print (x);  } while (0)

// _______________________________________________________________________


constexpr uint8_t row_N   = 4;
constexpr uint8_t col_N   = 6;
constexpr int16_t curSym  = 219;

int nav_btns[5] = {2,8,4,6,5};

KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);

uint8_t rowPins[row_N] = {
  14, // -> KP 2   ( R1 )
  11, // -> KP 5   ( R2 )
  9,  // -> KP 7   ( R3 )
  8   // -> KP 8   ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1   ( C1 )
  13, // -> KP 3   ( C2 )
  12, // -> KP 4   ( C3 )
  10, // -> KP 6   ( C4 )
  7,  // -> KP 9   ( C5 )
  6   // -> KP 10  ( C6 )
}; 



int16_t keypad_alpha[9][4] = {
  {230, 248, 229},      //  KP1 -> æ,ø,å
  {97,  98,  99},       //  KP2 -> a,b,c
  {100, 101, 102},      //  KP3 -> d,e,f
  {103, 104, 105},      //  KP4 -> g,h,i
  {106, 107, 108},      //  KP5 -> j,k,l
  {109, 110, 111},      //  KP6 -> m,n,o
  {112, 113, 114, 115}, //  KP7 -> p,q,r,s
  {116, 117, 118},      //  KP8 -> t,u,v
  {119, 120, 121, 122}, //  KP9 -> x,y,z
};




// - - - - - - - - - - - -

enum : uint8_t {
  BTN_UP    = 2,
  BTN_LEFT  = 4,
  BTN_RIGHT = 6,
  BTN_DOWN  = 8,
  BTN_ENTER = 5,
  BTN_BACK  = 10
};

enum : uint16_t {
  OFF  = 0,
  ON   = PWM_DEFAULT,
  FULL = PWM_MAX
};

enum : uint8_t {
  LCD_PRINT,
  LCD_PRINTLN,
  LCD_WRITE
};

enum : uint8_t {
  UI_READ,
  UI_WRITE,
  UI_STDIN,
  UI_STDIN_DONE
};

// - - - - - - - - - - - -

struct KeyMap {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  const char* t9_page;
};

/*
 01-09    + HOLD:10(*) = reverse paging in T9
 01-09+11 + HOLD:12(#) = 
 */


KeyMap matrix[4][3] = {
  {
    { colPins[0], 1, 0, ".,!?:;_-+=*/'\"%$&#@(){}[]"},
    { colPins[2], 2, 0, "abcABC"},
    { colPins[1], 3, 0, "defDEF"},
  },
  {
    { colPins[0], 4, 0, "ghiGHI"},
    { colPins[2], 5, 0, "jklJKL"},
    { colPins[3], 6, 0, "mnoMNO"}
  },
  {
    { colPins[0], 7, 0, "pqrsPQRS"},
    { colPins[2], 8, 0, "tuvTUV"},
    { colPins[3], 9, 0, "wxyzWXYZ"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " "},
    { colPins[4], 12, 0, ""}
  },
};

// - - - - - - - - - - - -

struct PWM_struct {
  uint16_t      pin;
  bool          state;
  float         clk;
  uint16_t      pwm;
  uint16_t      pwm_prv;
  uint16_t      slice;
  unsigned long timer;
};

PWM_struct bgl_data = { 
  BGL_PIN, 
  false,
  519.5, 
  ON, 
  FULL, 
  0, 
  0
};

/* ------------------------------------------------- */

char           buf_input[256];
uint8_t        buf_index;
uint8_t        ui_mode;
uint16_t       bgl_fadeout;
unsigned long  bgl_timeout;
unsigned long  kp_last_press, T;
bool           kp_lmod;
bool           kp_rmod;
bool           bootsel_held;
bool           menu_init;
String         buf;
//bool           activeState = true;

/* ------------------------------------------------- */

// Function declarations
void Menu();
void GLCD_footer(bool preserve = false);
void GLCD_indent(uint8_t margin, bool disp=false, char c=' ');
//void GLCD_overwrite(const char* s, int16_t ypos, const char* prefix);

bool isNavKey(uint8_t key);
unsigned int rawKeyPad(bool break_on_press = true);
// ___________________________________________________


bool isNavKey(uint8_t key) {
  for (uint8_t i=0; i<ArrLen(nav_btns); i++) {
    if (key == nav_btns[i])
      return true;
  }
  return false;
}

// ------------------------------------------------- 

bool GLCD_setBgl(uint16_t pwm, unsigned long t = BGL_TIMEOUT, bool state=false, PWM_struct &pwm_data = bgl_data) {
  pwm_data.pwm     =  pwm;
  pwm_data.timer   =  ((state && !pwm_data.state) || (pwm == ON)) ? millis() : pwm_data.timer;
  pwm_data.state   =  ((pwm == ON) || (state)) ? true : false;
   
  pwm_set_gpio_level( pwm_data.pin,   pwm            );
  pwm_set_enabled(    pwm_data.slice, (bool)(pwm > 0));

  bgl_timeout = ((pwm == ON) && pwm_data.state) ? t : bgl_timeout;

  return pwm_data.state;
}
// ------------------------------------------------- 

void GLCD_getBgl(unsigned long t = bgl_timeout, PWM_struct &pwm_data = bgl_data) {
  if ((millis()-bgl_data.timer) > t) {
    if (bgl_data.state || bgl_data.pwm > 0) {
      if (bgl_fadeout > 0)
        GLCD_setBgl( --bgl_data.pwm ),
        delay(bgl_fadeout);
      else
        GLCD_setBgl(OFF);
    }
    else
      GLCD_setBgl(OFF);
  }
}
/* ------------------------------------------------- */

bool PWM_Init(PWM_struct &pwm_data = bgl_data) {
  uint16_t pwm_max   =  255,
           pwm_pin   =  BGL_PIN,
           pwm_slice =  pwm_gpio_to_slice_num(pwm_pin);
           
  float    pwm_freq  =  1000.0,
           pwm_clk   =  (MCU_CLOCK/pwm_freq)/(float)(pwm_max+1);
           

  gpio_set_function(    pwm_pin,    GPIO_FUNC_PWM );

  pwm_set_clkdiv(       pwm_slice,  pwm_clk       );
  pwm_set_wrap(         pwm_slice,  pwm_max       );
  
  pwm_data.pin       =  pwm_pin;
  pwm_data.clk       =  pwm_clk;
  pwm_data.slice     =  pwm_slice;

  T = millis();

  return GLCD_setBgl(ON);
}
/* ------------------------------------------------- */

void GLCD_overflow(int maxY = 56) {
  if (display.getCursorY() > maxY)
    display.pushUp(8),
    display.setCursor(0, maxY);
}
// ---------------------------------------------------------------

void GLCD_animate(int del=20, int ripl=2) {
  int16_t i;
  unsigned long t = millis();
  
  display.clearDisplay();
  display.display();

  for(i = 0; i < max(display.width(),display.height())/2; i += 2) {
    if (i >= ripl)
      display.drawCircle( display.width()/2, display.height()/2, i-ripl, KS0108_OFF );
        
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_ON );    
    display.display();

    while ((millis()-t)<del) GLCD_getBgl();
    t = millis();
  }
  
  for(; i < (max(display.width(),display.height())/2) + (10+ripl); i += 2) {
    display.drawCircle( display.width()/2, display.height()/2, i-ripl, KS0108_OFF );
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_ON );
    display.display();
    delay(del);
  }
  delay(200);
}
/* ------------------------------------------------- */

bool GLCD_Init() {  
  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false ) 
    return false;

  display.display();                // display flash

  bgl_fadeout = (BGL_FADEOUT) ? BGL_FADEOUT : 0;
  bgl_timeout = BGL_TIMEOUT;
  
  GLCD_setBgl(ON, 30000);

  delay(3000);

  //display.setFont(&Picopixel);    // Set font, default = NULL
  display.clearDisplay();           // Clear the buffer
  display.setTextSize(1);           // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);  // Draw white text
  display.setCursor(0, 0);          // Start at top-left corner
  display.cp437(true);              // Use full 256 char 'Code Page 437' font
  display.setTextWrap(true);        // Toggle text wrap (else clip right).
  display.display();
  
  return true;
}
/* ------------------------------------------------- */ 

void GLCD_print(const char* txt, uint8_t mode=LCD_PRINT, bool preClear=false) {
  GLCD_setBgl(ON);

  if (preClear)
    display.clearDisplay(),
    display.setCursor(0,0);
  
  display.print(txt);

  if (mode == LCD_PRINTLN)
    display.print("\n");
  display.display();
}
/* ------------------------------------------------- */ 

template <typename T>
void GLCD_print(T txt, uint8_t mode, bool preClear) {
  GLCD_setBgl(ON);
  
  if (preClear)
    display.clearDisplay(),
    display.setCursor(0,0);
    
  switch(mode) {
    default:
    case 0:
      display.print(txt);
      break;
    case 1:
      display.println(txt);
      break;
    case 2:
      display.print(txt);
      break;
  }
  display.display();
}

/* ------------------------------------------------- */

void GLCD_indent(uint8_t margin, bool disp, char c) {
  if (margin==0) return;
  
  for (uint8_t n=0; n<margin; n++)  
    display.write(c);
    
  if (disp) 
    display.display();
}
// ---------------------------------------------------------

void GLCD_overwrite(const char* s, int16_t ypos=56, const char* prefix="> ") {
  int16_t y = display.getCursorY();
  display.setCursor(0, ypos);
  //display.setTextColor( (!display.isInverted()) ? KS0108_OFF : KS0108_ON );
  display.setTextColor(KS0108_OFF);
  
  for (int sz = 0; sz < 22; sz++) 
    display.write(219); 
  display.display();
  
  //display.setTextColor( (!display.isInverted()) ? KS0108_ON : KS0108_OFF );
  display.setTextColor(KS0108_ON);
  display.setCursor(0, ypos);
  
  if (prefix)
    display.print(prefix);
    
  display.print(s);
  display.display();
    
  display.setCursor(0, y);
}
/* ------------------------------------------------- */

void GLCD_footer(bool preserve) {
  display.setCursor(0, (8*7));
  display.print("> ");
  display.display();
}

/*
void GLCD_footer(bool preserve) {
  uint8_t xpos = display.getCursorX();
  uint8_t ypos = display.getCursorY();
  display.setCursor(0, (8*7));
  //display.setTextColor( (!display.isInverted()) ? KS0108_OFF : KS0108_ON );
  display.setTextColor(KS0108_OFF);
  
  for (uint8_t i = 0; i<display.width()/6; i++)
    display.write(219);
  display.display();
    
  display.setTextColor(KS0108_ON);
  display.print("> ");
  display.display();
  
  if (preserve)
    display.setCursor(xpos, ypos);
}
*/
/* ------------------------------------------------- */
void setBit(unsigned int &target, uint8_t x, bool set) {  
  if (set)
    target |= (1 << x);
  else
    target &= ~(1 << x);
    
  target &= 0b0000111111111111;
}
// -------------------------------------------------- */

bool getBit(unsigned int target, uint8_t x) {
  return (bool)(target & (1 << x));
}
// -------------------------------------------------- */

void t9_reset() {
  for (uint8_t x = 0; x < 4; x++) 
    for (uint8_t y = 0; y < 3; y++) 
      matrix[x][y].index = 0;
}
// ---------------------------------------------------------------

void clearBuf() {
  memset(buf_input, 0, sizeof(buf_input));
  buf_index = 0;
  t9_reset();
}
// ---------------------------------------------------------------


unsigned int rawKeyPad(bool break_on_press) {
  unsigned int binary = 0b1000000000000000;
  
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      uint8_t col_pin = matrix[row][col].pin;
      uint8_t key = matrix[row][col].key;
      
      if (digitalRead(col_pin) == LOW) {
        setBit(binary, key-1, 1);
        
        if (break_on_press) {
          digitalWrite(rowPins[row], HIGH);
          return binary;
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  delay(KP_DEBOUNCE);
  return binary;
}
/* ------------------------------------------------- */

KeyMap *readKeyPad() {
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      uint8_t col_pin = matrix[row][col].pin;
      uint8_t key = matrix[row][col].key;  
      
      if (digitalRead(col_pin) == LOW) {
        if (millis()-kp_last_press) {
          while (digitalRead(col_pin) == LOW); 
          digitalWrite(rowPins[row], HIGH);
          kp_last_press = millis();
          
          return &(matrix[row][col]);
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  return nullptr;
}
/* ------------------------------------------------- */

bool checkInput() {
  uint8_t key, index;
  char c;
  
  KeyMap *obj = readKeyPad();
  
  if (obj != NULL) {
    key = obj->key;

    if (obj->index < 1) {
      if (buf_input[buf_index] != '\0')
        buf_index = (key != 10 && key != 12) ? buf_index + 1 : buf_index;
      t9_reset();
    } 
    else {
      if (key > 10)  
        buf_index++;
    }
    
    if (key == 10) {
      buf_input[buf_index] = '\0';
      buf_index = (buf_index > 0) ? buf_index - 1 : 0;
    }
      
    index = obj->index % strlen(obj->t9_page);
    
    if (strlen(obj->t9_page) > 0) {
      buf_input[buf_index] = obj->t9_page[index];
      obj->index++;
    }
    return true;
  }
  return false;
}

// ---------------------------------------------------------------

bool checkSerial(bool discard=false) {
  if (!Serial.available()) 
    return false;
  else if (discard) 
    Serial.readStringUntil('\n');
  return true;
}

// ---------------------------------------------------------------

void checkHeader(String header) {
  delay(1);
}
// ----------------------------------------

namespace tree {
  struct  TreeNode *node_active, *node_select;
  uint8_t level, sub_index;
  
  // ----------------------------------------

  uint8_t getId(uint16_t val = node_select->id, uint8_t lvl = level) {
    return ((val >> (lvl * 4)) & 0xF);
  }
  // -----------------------------------------

  void showTitle(const char* title, char fillChar = '-') {
    char screenBuf[22]; 
    int i, n = 0;
    int leftPad;
    
    memset(screenBuf, '\0', sizeof(screenBuf));

    for (i = 0; i < strlen(title); i++)
      if (isprint(title[i]))
        screenBuf[n++] = (isalpha(title[i])) ? toupper(title[i]) : title[i];

    leftPad = (((display.width()-1)/6) - n) / 2;
    
    for (i = 0; i < leftPad; i++)
      display.write(fillChar);

    display.print(screenBuf);
    
    for (n = (leftPad + n); n < ((display.width()-1)/6); n++)
      display.write(fillChar);
    display.println("\n");

    display.drawLine(0, 9, display.width()-1, 9, KS0108_ON);
    display.drawLine(0, 11, display.width()-1, 11, KS0108_ON);
  }
  // ----------------------------------------
    
  void Render(struct TreeNode* node = node_active) {
    struct TreeNode* child = NULL;

    if (node && node->sub_count) {
      GLCD_setBgl(ON);
      display.clearDisplay();
      
      showTitle(node_active->label, 176);
      
      for (uint8_t i = 0; i < node->sub_count; i++) {
        display.setTextColor(KS0108_ON);
        child = node->children[i];
      
        if (!child) continue;
        if (node_select->id == child->id)
          display.setTextColor(KS0108_OFF, KS0108_ON);

        display.print( (child->sub_count) ? "/ " : "* ");
        display.println(child->label);
        display.display();
        
        display.setTextColor(KS0108_ON);
      }
    }
  }
  // ----------------------------------------

  void Select(uint8_t index, struct TreeNode* node = node_active) {
    node_active = node;
    node_select = (node->sub_count == 0) ? node : node->children[0];
    sub_index = (node->sub_count == 0) ? 1 : sub_index;
    
    if (node->sub_count == 0) 
      return;

    index = ((index > node->sub_count) ? 1 : ((index < 1) ? node->sub_count : index));
    node_select = node->children[index-1];
    sub_index = index;
    
    Render(node);
  }
  // ----------------------------------------
  
  bool Init(struct TreeNode* node = &(menuTree), bool initialized = true) {
    if (!node) 
      return initialized;
    
    if (!initialized) {
      if (node-> id != 0x0001) 
        node = &(menuTree);
        
      level = sub_index = 1;
      initialized = true;

      Select(1, node);
    }

    if (node->sub_count) {
      for (uint8_t i = 0; i < node->sub_count; i++) {
        (*node->children[i]).parent = node;
        Init( node->children[i], initialized);
      }  
    }

    return initialized;
  }
  // ----------------------------------------

  void CallFunc(void(*func)(void *), void *param){ 
    func(param);
  }
  // ----------------------------------------

  void GoTo(struct TreeNode* node) {
    if (!node) 
      return;
      
    if (node->id < node_active->id) {
      Select(1, node);
      level = (level > 1) ? level-1 : 1;
    }
    else {
      if (node->sub_count) {
        Select(1, node);
        level++;
      }
      else if (node->action != NULL) {
        node->action();
      }      
    }
  }
  // ----------------------------------------

  void Navigate(uint8_t key) {
    if (key == BTN_UP) {
      sub_index--;
      Select(sub_index);
    }
    else
    if (key == BTN_DOWN) {
      sub_index++;
      Select(sub_index);
    }
    else
    if (key == BTN_ENTER) {
      GoTo(node_select);
    }
    else
    if (key == BTN_BACK) {
      GoTo(node_active->parent);
    }     
  }

  // ----------------------------------------

  void Debug(struct TreeNode* node, int depth) {
    if(!node) return;
  
    if ((*node_active).id == node->id)
      display.setTextColor(KS0108_OFF, KS0108_ON); 
    else
      display.setTextColor(KS0108_ON); 
      
    display.print((node->parent != NULL) ? (*node->parent).label : "null");
    display.print("->");
    display.println(node->label);
    display.display();
    display.setTextColor(KS0108_ON); 
    delay(250); // REM
    
    if (node->sub_count) {
      for (uint8_t i = 0; i < node->sub_count; i++) {
        if(node->children[i])
          Debug(node->children[i], depth+1);
      }
    }
  }
}
// ---------------------------------------------------------

void printFuncName(const char* id, uint8_t ypos = 7) {
  //display.setCursor(0, (8*7));
  display.print("calling '");
  display.print(id);
  display.print("()'");
  display.display();
}

void myFun() {
  GLCD_footer(false);
  printFuncName(__func__);
  
  for (int i=0; i<5; i++)
    digitalWrite(28, !digitalRead(28)),
    delay(500);
    
  digitalWrite(28, HIGH);
}

/*
int16_t alpha[] = {
  97,   // a
  98,   // b
  99,   // c
  100,  // d
  101,  // e
  102,  // f
  103,  // g
  104,  // h
  105,  // i
  106,  // j
  107,  // k
  108,  // l
  109,  // m
  110,  // n
  111,  // o
  112,  // p
  113,  // q
  114,  // r
  115,  // s
  116,  // t
  117,  // u
  118,  // v
  119,  // w
  120,  // x
  121,  // y
  122,  // z
  230,  // æ
  248,  // ø
  229,  // å
};
 */
    
    /*
    if (node && node->sub_count) {
      showTitle(node_active->label);

      GLCD_setBgl(ON);
      display.clearDisplay();

      display.print("...");
      display.print(screenBuf);
      display.println("...");
      display.display();

      display.printf("%*s\n", 
        (((leftPad/2)+(strlen(screenBuf)/2))+(strlen(screenBuf)/2)), 
        "oi mate");
      display.display();
      
    }
    */
    
    /*
    const char* title = node_active->label;

    if (node && node->sub_count) {
      GLCD_setBgl(ON);
      display.clearDisplay();

      int leftPad = (((display.width()/6)/2)  + (strlen(title)/2));

      for (int i = 0; i < leftPad; i++)
        display.write(' ');
        
      for (int i = 0; i < strlen(title); i++)
        if (isprint(title[i]))
          display.write( isalpha(title[i]) ? toupper(title[i]) : title[i] );
                
      display.display();
      delay(1000);
      
      display.printf("%*s\n",
        ((display.width()/6)/2 + (strlen(node_active->label)/2)),
        node_active->label );
      
      for (uint8_t i = 0; i < node->sub_count; i++) {
        display.setTextColor(KS0108_ON);
        child = node->children[i];
        
        if (!child) continue;
        if (node_select->id == child->id)
          display.setTextColor(KS0108_OFF, KS0108_ON);

        //int space = 20;
        //display.printf("%*s\n", (10+(strlen(child->label)/2)), child->label);

        //display.printf("%*s", 5);
        //display.print( (child->sub_count) ? "/ " : "* ");        
        
        display.println(child->label);
        display.display();
      }
    }
    */
