#include <Adafruit_TinyUSB.h>
#include "ramdisk.h"
#include <Adafruit_GFX.h>
#include <KS0108_GLCD.h>
#include <TinyUSB_Mouse_and_Keyboard.h>

#include "tree.h"
#include "graphics.h"
#include "include.h"

typedef enum : int8_t {
  OTHER = -1,
  BOOTING,
  BOOTED,
  MENU_ON,
  STDIN_MSC,
  STDIN_SER,
  READER,
  TXT_INPUT,
  AWAIT_RESP
} state_enum;

struct _systatus {
  state_enum  id;
  const char *txt;
};
_systatus sys_states[] = {
  {BOOTING,   "Booting."},
  {BOOTED,    "System ready."},
  {MENU_ON,   "Menu launched."},
  {STDIN_MSC, "Reading MSC in."},
  {STDIN_SER, "Reading Serial in."},
  {READER,    "File reader."},
  {TXT_INPUT, "Text input."},
  {AWAIT_RESP, "Awaiting response."}
};

_systatus
sys_status_prev,
sys_status = sys_states[BOOTING];

// ----------------------------------------

void setState(int8_t state, const char* custom_txt = NULL) {
  sys_status_prev = sys_status;
  sys_status = sys_states[state];
}

state_enum getState(_systatus* state = &sys_status) {
  return state->id;
}
// ----------------------------------------

void GLCD_msgBox(const char* msg, bool preClear = true, int16_t divi = 2, int16_t padd = 20) {
  int16_t offs = display.width()/divi;
  int16_t x1, y1, x2, y2;
  int16_t N = (offs/divi)+padd;
  
  if (preClear)
    display.clearDisplay();

  for(int16_t i=2; i<N; i++) {
    x1 = offs-i;
    y1 = (offs/divi)-(i/divi);
    x2 = i*divi;
    y2 = i;
    
    display.fillRect(x1, y1, x2, y2, KS0108_OFF);
    display.drawRect(x1, y1, x2, y2, KS0108_ON);
    display.display();
    delay(1);  
  }  
  delay(100);

  for(int i=0; i<(N*2)-2; i+=4) {
    display.drawLine(x1, y1+10, x1+i, y1+10, 1);
    display.display(); // Update screen with each newly-drawn line
    delay(1);
  }

  display.setCursor(x1+40, y1+2);
  display.print("MSG");
  display.display();
  delay(100);
  
  x1 = x1+4;
  y1 = y1+12;
  
  display.setTextWrap(false);
  display.setCursor(x1,y1);
  
  for (int i=0; i<strlen(msg); i++) {
    if ((display.getCursorX() > (N*2)+6) || (msg[i] == '\n' || msg[i] == '\r')) 
      display.setCursor(x1, y1=(y1+8));
    
    if (msg[i] != '\n' && msg[i] != '\r')
      display.write(msg[i]);
  }
  display.display();
  display.setTextWrap(true);
  delay(1000);
}
// ----------------------------------------

void showBitmap(const uint8_t *bmp, uint16_t del = 3000) {
  display.clearDisplay();
  display.drawBitmap(
    (display.width()  - BMP_WIDTH ) / 2,
    (display.height() - BMP_HEIGHT) / 2,
    bmp, 
    BMP_WIDTH, 
    BMP_HEIGHT, 
    1
  );
  
  display.display();
  delay(del);
}
// ---------------------------------------------------------

void setup() { 
  usb_msc_enable(true);
  usb_msc.begin();  
  
  Serial.begin(9600);
  Keyboard.begin();
  Keyboard.releaseAll(); 
  
  pinMode( BGL_PIN, OUTPUT_12MA );
  digitalWrite( BGL_PIN, HIGH );

  GLCD_Init();
  //PWM_Init();

  setState(BOOTING);
  
  // - - - - - - - - - - - - - - - - - 
  
  for (int i = 0; i < row_N; i++)
    pinMode( rowPins[i],      OUTPUT ),
    digitalWrite( rowPins[i], HIGH   ),
    
    gpio_set_slew_rate( 
      rowPins[i], GPIO_SLEW_RATE_SLOW);
  
  for (int i = 0; i < col_N; i++)
    pinMode( colPins[i], INPUT_PULLUP );
  
  // - - - - - - - - - - - - - - - - - 

  while (!msc_ready) yield();
  setState(BOOTED);
  
  while (!zero_booted) {
    if (Serial.available() || BOOTSEL) break;
    if ((millis()-T) > 1000) 
      GLCD_animate(30,6),
      T = millis();
    GLCD_getBgl();
    break; // REM
  }
  while (Serial.available()) 
    Serial.read();

  zero_booted = true;
  clearBuf();

  //showBitmap(cat_bmp);
  textInput();
  //Menu();  
}
// ---------------------------------------------------------

void Menu() {
  ui_mode = UI_READ;
  setState(MENU_ON);
  
  if (!menu_init) 
    //GLCD_setBgl(ON, 30000),
    menu_init = tree::Init(&menuTree, menu_init);

  tree::Render();
}
// ---------------------------------------------------------

void textInput() {
  setState(TXT_INPUT);
  ui_mode = UI_WRITE;
  clearBuf();
  mod_mode = 0;
  GLCD_print("> ", LCD_PRINT, true);
  delay(2000);
}

// ---------------------------------------------------------

void Reader() {
  const char  *txt_buf;
  uint16_t    last, index;
  uint8_t     key;
  bool        valid, curstate;
  KeyMap      *KP;
  
  txt_buf = (sys_status.id == STDIN_MSC) ? stdin_buf : ((sys_status.id == STDIN_SER) ? buf_input : nullptr);
  
  if (txt_buf != NULL && strlen(txt_buf)) {
    setState(READER);
    
    while (1) {
      curstate = false;
      index    = 0;
      
      display.clearDisplay();
      display.display();
      delay(100);
      
      while (index < strlen(txt_buf)-1) {
        display.write(txt_buf[index++]);
        display.display();
        delay(10);
        
        if (display.getCursorY() > 56) {
          last  = index;
          valid = false;
          key   = 0;
          
          while (!valid) {
            KP = readKeyPad();
            if (KP != NULL) {
              key = KP->key;
              
              if (key == 7 || key == 10 || key == 12) {
                valid = false;
                break;
              }
              else
                valid = true;
            }
            if ( !(millis() % 200) ) 
              GLCD_blinkCursor((curstate = !curstate), (20*6), 56, 25);
          }
          GLCD_blinkCursor(false, (20*6), 56, 25);
          
          if (KP != NULL) {
            if (!valid) return;
            
            display.pushUp(8);
            display.setCursor(0, 56);
            display.display();
            delay(100);
          }
        }
      }
      GLCD_blinkCursor(true, (20*6), 56, 19);
      
      while ((readKeyPad() == NULL) && !BOOTSEL);
      GLCD_msgBox("\n  READ AGAIN?\n\n    PRESS 5", false);
      
      while (1) {
        KP = readKeyPad();
        if (KP != NULL) break;
      }
      if (KP->key != 5) return;
    }    
  }
}
// ---------------------------------------------------------

void awaitResponse() {
  showBitmap(cat_bmp);
  setState(AWAIT_RESP);
  ui_mode = UI_READ;
  await_response = true;
}
// ---------------------------------------------------------

void sendCommand() {
  KeyMap      *KP;
  
  GLCD_msgBox("\n  SEND COMMAND?\n\n    PRESS 5", false);
  while (1) {
    KP = readKeyPad();
    if (KP != NULL) break;
  }
  if (KP->key == 5) {
    delay(1000);
    Keyboard.println(buf_input);
    awaitResponse();
  }
  else
    Menu();
}
// ---------------------------------------------------------

void loop() {
  if (ui_mode == UI_WRITE) {
    if (checkInput()) {
      display.clearDisplay();
      display.print("> ");
      display.println(buf_input);
      display.println();
      display.print("mod: "); 
      display.println(mod_mode);
      display.display();
      delay(30);
    }
    else if (BOOTSEL) {
      kp_last_press = millis();
      while (BOOTSEL) {
        if ((millis()-kp_last_press) > 2000) 
          break;
      }
      if ((millis()-kp_last_press) > 2000) 
        Menu();
      else
        sendCommand();
    }
  }
  
  else if (ui_mode == UI_READ) {
    
    if (getState() == MENU_ON) {
      KeyMap *KP = readKeyPad();  
      if (KP != NULL)
        tree::Navigate(KP->key);
    }
    
    if (stdin_available) {
      setState(STDIN_MSC);
      
      if (await_response)
        Reader(),
        await_response = false;
        
      stdin_available = false;
      Menu();
    }
    else if (Serial.available()) {
      if (Serial.find("#STDIN\n")) {
        int n = 0; T = millis();
        
        memset(buf_input, 0, sizeof(buf_input));
        serial_busy = true;
        setState(STDIN_SER);
        
        while (serial_busy) {
          if (Serial.available() > 0) { 
            char *eof, c = Serial.read();
            buf_input[n++] = c;
            eof = strstr(buf_input, "#EOF\n");
            T = millis();
            
            if (eof) {
              *eof = '\0';
              serial_busy = false;
            }
          }
          else if ((millis()-T) > 5000)
            serial_busy = false;
        }
        if (await_response)
          Reader(),
          await_response = false;
      }
      Menu();
    }
  }
}

/*
void loop() {
  unsigned int btn_map;
  
  if (BOOTSEL) { kp_last_press = millis();
    while (BOOTSEL) { 
      if ((millis()-kp_last_press)>2000) {
        GLCD_animate(30,12);
        bootsel_held = true;
      }
    }
    if (bootsel_held) {
      ui_mode = (ui_mode == UI_WRITE) ? UI_READ : UI_WRITE;
      GLCD_print( ((ui_mode == UI_READ) ? "UI: READ MODE" : "UI: WRITE MODE"), LCD_PRINTLN, true);
      bootsel_held = false;
    }
  }
  else if (Serial.available() > 0) {
    buf = Serial.readStringUntil('\n'); buf.trim();
    if (!zero_booted) {
      zero_booted = true;
      return;
    }
    
    if (buf.length()) {
      GLCD_print(buf.c_str(), LCD_PRINTLN, true);
      
      if (buf == "reset") {
        pwm_set_gpio_level( bgl_data.pin, 0 );
        pwm_set_enabled( bgl_data.slice, false);
        delay(1000);
        display.resetDisplay();
        delay(1000);
        pwm_set_gpio_level( bgl_data.pin, 255 );
        pwm_set_enabled( bgl_data.slice, true);
        
      }
    }    
  }
  else if (ui_mode == UI_STDIN || ui_mode == UI_STDIN_DONE) { T = millis();
    GLCD_print("Waiting for STDIN..", LCD_PRINTLN, true);
    while (ui_mode == UI_STDIN) {
      if ((millis()-T)>1000)
        GLCD_animate(30,8),
        T = millis();   
    }
    delay(1000);
    ui_mode = UI_READ;

    display.clearDisplay();
    
    for (uint32_t i=0; i<strlen(stdin_buf); i++) {
      if (display.getCursorY() > 56) {
        while (!BOOTSEL) yield();
        display.clearDisplay();
      }
      display.write(stdin_buf[i]);
      display.display();
      delay(5);
    } 
  }
  else if (ui_mode == UI_WRITE) {
    btn_map = rawKeyPad();
    bool mod_pressed = (btn_map & (1 << 11) ) ? true : false;
  }
  else if (ui_mode == UI_READ) {
    btn_map = rawKeyPad();
    int i=0;

    if (btn_map) {
      display.clearDisplay();
      for (int& btn : nav_btns) {
        if (btn_map & (1 << btn-1) ) {
          display.write((int)btn + '0'); display.write(' ');
          display.display();
          //break;
        }
      }
    }
  }
  GLCD_getBgl();
}
*/
/*
void loop() {
  if (BOOTSEL) {
    while (BOOTSEL);

    if (strlen(buf_input) > 0) {
      //Keyboard.print("run ");
      //Keyboard.println(buf_input);
      Serial.println(buf_input);
      display.clearDisplay();
      display.setCursor(0,0);
      display.print("> "); 
      display.display();
      
      clearBuf();
    }
  }
  else {
    if (checkInput()) {

      display.clearDisplay();
      display.setCursor(0,0);
      display.print("> ");
      
      if (ui_mode == UI_STDIN_DONE) {
        ui_mode = UI_WRITE;
        clearBuf();
      }
      else {
        display.print(buf_input);
        display.write(curSym);
        display.write('\n');
        display.display();
      }
      delay(30);
    }
  }
  GLCD_getBgl();
}
*/
/*
void loop() {
  if (Serial.available() > 0) {
    buf = Serial.readStringUntil('\n'); buf.trim();
    
    if (!zero_booted) {
      zero_booted = true;
      return;
    }
    
    if (buf.length())
      GLCD_print(buf, LCD_PRINTLN, true);
      

    if (buf == "#STDIN") {
        display.clearDisplay();
        display.setCursor(0,0);
        display.display();
        
        file = LittleFS.open("/stdin.txt", "w+");
        if (file) 
          file.seek(0),
          file.printf(""),
          file.close();
        
        ui_mode = UI_STDIN;
        return;
    }
    else if ((buf.substring(0, buf.indexOf(' ')) == "#HEAD") || (buf.startsWith("#HEAD"))) {
        checkHeader( buf.substring(buf.indexOf(' ')+1, buf.length()) );
        return;
    }
    else if (buf == "#EOF") {
      display.clearDisplay();
      display.setCursor(0,0);
      
      ui_mode = UI_STDIN_DONE;
      return;
    }
    // - - - - - - - - - - - -

    if (ui_mode == UI_STDIN) {
      
      file = LittleFS.open("/stdin.txt", "a");

      if (file) 
        file.printf("%s\n", buf.c_str()),
        file.close();
      
      delay(READ_DEL);
    }
  }
}
*/
// _________________________________________________________
