#pragma once

#include <pico/stdlib.h>
#include <hardware/pwm.h>

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#define ArrLen(x)         (sizeof(x)/sizeof(x[0]))
#define bit2byte(x,pos)   (((x) >> ((pos) * 4)) & 0xF)

#define LED_PIN           LED_BUILTIN
#define MCU_CLOCK         (133000000.0)
#define RESP_TIMEOUT      30000UL
#define BGL_TIMEOUT       10000UL
#define READ_DEL          250
#define LCD_MAX_CHARS     168
#define SERIAL_BUF_SIZE   255
#define HEADER_SIZE       64
#define BGL_PIN           28
#define KP_DEBOUNCE       30
#define TIME_REQUEST      7
#define DEVICE_VID        0x0920
#define DEVICE_PID        0x1986
#define MANUFACTURER      "data_dogenigt"
#define DEVICE_NAME       "LoPi-Phone v1.0"

// _______________________________________________________________________

constexpr uint8_t row_N   = 4;
constexpr uint8_t col_N   = 6;
constexpr int16_t curSym  = 219;
constexpr char ae = -26; // \xE6 \u00E6
constexpr char oe = -8;  // \xF8 \u00F8
constexpr char aa = -27; // \xE5 \u00E5

KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);

uint8_t rowPins[row_N] = {
  14, // -> KP 2   ( R1 )
  11, // -> KP 5   ( R2 )
  9,  // -> KP 7   ( R3 )
  8   // -> KP 8   ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1   ( C1 )
  13, // -> KP 3   ( C2 )
  12, // -> KP 4   ( C3 )
  10, // -> KP 6   ( C4 )
  7,  // -> KP 9   ( C5 )
  6   // -> KP 10  ( C6 )
};

// - - - - - - - - - - - -

enum : uint8_t {
  BTN_UP    = 2,
  BTN_LEFT  = 4,
  BTN_RIGHT = 6,
  BTN_DOWN  = 8,
  BTN_ENTER = 5,
  BTN_BACK  = 10
};

enum : uint8_t {
  LCD_PRINT,
  LCD_PRINTLN,
  LCD_WRITE
};

enum : uint8_t {
  UI_READ,
  UI_WRITE,
  UI_STDIN,
  UI_STDIN_DONE
};

enum : uint8_t {
  T9_MODE_ALPHA,
  T9_MODE_CAPS,
  T9_MODE_NUM,
  T9_MODES_N
};

enum : uint8_t { 
  CMD_HID, 
  CMD_SER 
};

enum : bool { 
  OFF = false,
  ON  = true
};

enum : int { 
  KP_MOD = -1 
};

typedef enum : int8_t {
  OTHER = -1,
  BOOTING,
  BOOTED,
  HOME_ON,
  MENU_ON,
  STDIN_MSC,
  STDIN_SER,
  READER,
  TXT_INPUT,
  AWAIT_RESP
} state_enum;

// - - - - - - - - - - - -

struct KeyMap {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  char        t9_page[5];
};
KeyMap matrix[4][3] = {
  {
    { colPins[0], 1, 0, {  ae,  oe,  aa,  0,  0 }},
    { colPins[2], 2, 0, { 'a', 'b', 'c',  0,  0 }},
    { colPins[1], 3, 0, { 'd', 'e', 'f',  0,  0 }},
  },
  {
    { colPins[0], 4, 0, { 'g', 'h', 'i',  0,  0 }},
    { colPins[2], 5, 0, { 'j', 'k', 'l',  0,  0 }},
    { colPins[3], 6, 0, { 'm', 'n', 'o',  0,  0 }},
  },
  {
    { colPins[0], 7, 0, { 'p', 'q', 'r', 's', 0 }},
    { colPins[2], 8, 0, { 't', 'u', 'v',  0,  0 }},
    { colPins[3], 9, 0, { 'w', 'x', 'y', 'z', 0 }},
  },
  {
    { colPins[2], 10, 0, { 0,   0,   0,   0,  0 }},
    { colPins[5], 11, 0, {' ',  0,   0,   0,  0 }},
    { colPins[4], 12, 0, { 0,   0,   0,   0,  0 }},
  },
};

// - - - - - - - - - - - -

struct _systatus {
  state_enum  id;
  const char *txt;
};
_systatus sys_states[] = {
  {BOOTING,   "Booting."},
  {BOOTED,    "System ready."},
  {HOME_ON,   "Home screen."},
  {MENU_ON,   "Menu launched."},
  {STDIN_MSC, "Reading MSC in."},
  {STDIN_SER, "Reading Serial in."},
  {READER,    "File reader."},
  {TXT_INPUT, "Text input."},
  {AWAIT_RESP, "Awaiting response."}
};
_systatus
sys_status_prev,
sys_status = sys_states[BOOTING];

// - - - - - - - - - - - -

struct t9_mode_t {
  const char* label;
  uint8_t     enum_t;
  int16_t     icon;
}
KP_t9_modes[3] = {
  {"alpha_std",  T9_MODE_ALPHA, 42},
  {"alpha_caps", T9_MODE_CAPS,  94},
  {"numeric",    T9_MODE_NUM,   241},
};

t9_mode_t      KP_t9_mode;
uint8_t        t9_mod_mode;

struct _homeScreen {
  char name[5];
  char content[12];
} homeScreen[7] = {
  {"date","-"},
  {"time","-"},
  {"sun", "-"},
  {"day", "-"},
  {"moon","-"},
  {"temp","-"},
  {"ip","-"},
};

/* ------------------------------------------------- */

const char* months[12] = {
  "Jan","Feb","Mar",
  "Apr","May","Jun",
  "Jul","Aug","Sep",
  "Oct","Nov","Dec"
};

/* ------------------------------------------------- */
 
char           buf_input[LCD_MAX_CHARS]; // (21*8 = 168)
char           serial_buf[SERIAL_BUF_SIZE];
char           header[HEADER_SIZE];
char           hex_str[2]; 
uint8_t        buf_index;
uint8_t        ui_mode;
uint8_t        cmd_type_override;
unsigned long  bgl_timeout, bgl_timer;
unsigned long  resp_timer;
unsigned long  kp_last_press, T;
unsigned long  screenUpdate;
unsigned int   btn_map;
bool           bgl_state;
bool           bootsel_held;
bool           menu_init;
bool           await_response;
bool           serial_busy;
String         buf, dateStr, timeStr;

/* ------------------------------------------------- */

// Function declarations
void Home();
void Menu();
void sysTick();
void clock_tick();
void readSerialData();
void printTopBar(const char* title = nullptr);
void GLCD_indent(uint8_t margin, bool disp = false, char c = ' ');

// ___________________________________________________

time_t requestSync() {
  Serial.write(TIME_REQUEST);
  return 0; 
}
// ----------------------------------------

void setState(int8_t state, const char* custom_txt = NULL) {
  sys_status_prev = sys_status;
  sys_status = sys_states[state];
}
// ----------------------------------------

state_enum getState(_systatus* state = &sys_status) {
  return state->id;
}
// ----------------------------------------

char* hex2char(unsigned char x, char hex[] = hex_str) {
  hex[0] = (char)x;
  hex[1] = '\0';
  return hex;
}
// -------------------------------------------------

int16_t ascii2lcd(int16_t input) {
  switch(input) {
     case 230: return 145;  //  æ
     case 248: return 236;  //  ø
     case 229: return 134;  //  å
     case 198: return 146;  //  Æ
     case 216: return 237;  //  Ø
     case 197: return 143;  //  Å
     default: return input; // (*)
  }
}
// -------------------------------------------------

void BGL_set(bool state) {
  bgl_state = state;
  digitalWrite(BGL_PIN, state);
  bgl_timer = (state) ? millis() : bgl_timeout+1;
}

void GLCD_blinkCursor(bool state, int16_t x=0, int16_t y=56, int16_t c=219) {
  display.setCursor(x,y);
  display.setTextColor((state)? KS0108_ON : KS0108_OFF);
  display.write(c);
  display.display();
  display.setTextColor(KS0108_ON);
}
// ---------------------------------------------------------

void GLCD_overflow(int maxY = 56) {
  if (display.getCursorY() > maxY)
    display.pushUp(8),
    display.setCursor(0, maxY);
}
// ---------------------------------------------------------------

void GLCD_anim() {
  int16_t i, ripl = 2;
  unsigned long t = millis();

  //BGL_set(ON);
  display.clearDisplay();
  
  for (i = 0; i < max(display.width(), display.height())/2; i += 2) {
    if ((!zero_booted) && (Serial.available() || stdin_available)) break;
    display.drawBitmap((display.width()-BMP_WIDTH)/2, (display.height()-BMP_HEIGHT)/2, cat_bmp, BMP_WIDTH, BMP_HEIGHT, 1); 
    
    if (i >= ripl)
      display.drawCircle( display.width()/2, display.height()/2, (i-ripl), KS0108_OFF );
    
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_OFF );
    display.display();
    delay(30);
  }

  for (; i < (max(display.width(), display.height())/2)+(10 +ripl); i += 2) {
    if ((!zero_booted) && (Serial.available() || stdin_available)) break;
    display.drawBitmap((display.width()-BMP_WIDTH)/2, (display.height()-BMP_HEIGHT)/2, cat_bmp, BMP_WIDTH, BMP_HEIGHT, 1);  
    display.drawCircle( display.width()/2, display.height()/2, (i-ripl), KS0108_OFF );
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_OFF );
    display.display();
    delay(30);
  }   
}
// ---------------------------------------------------------------

bool GLCD_Init() {
  bgl_timeout = BGL_TIMEOUT;

  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false )
    return false;

  display.display();                // display flash
  delay(3000);
  BGL_set(ON);

  display.setFont(NULL);
  display.clearDisplay();           // Clear the buffer
  display.setTextSize(1);           // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);  // Draw white text
  display.setCursor(0, 0);          // Start at top-left corner
  display.cp437(true);              // Use full 256 char 'Code Page 437' font
  display.setTextWrap(true);        // Toggle text wrap (else clip right).
  display.display();

  return true;
}
/* ------------------------------------------------- */

void GLCD_print(const char* txt, uint8_t mode = LCD_PRINT, bool preClear = false) {
  BGL_set(ON);

  if (preClear)
    display.clearDisplay();

  display.print(txt);

  if (mode == LCD_PRINTLN)
    display.print("\n");

  display.display();
}
/* ------------------------------------------------- */

template <typename T>
void GLCD_print(T txt, uint8_t mode, bool preClear) {
  BGL_set(ON);

  if (preClear)
    display.clearDisplay(),
    display.setCursor(0, 0);

  switch (mode) {
    default:
    case 0:
      display.print(txt);
      break;
    case 1:
      display.println(txt);
      break;
    case 2:
      display.print(txt);
      break;
  }
  display.display();
}

/* ------------------------------------------------- */

void GLCD_indent(uint8_t margin, bool disp, char c) {
  if (margin == 0) return;

  BGL_set(ON);

  for (uint8_t n = 0; n < margin; n++)
    display.write(c);

  if (disp)
    display.display();
}
// ---------------------------------------------------------

void GLCD_overwrite(const char* s = " ", int16_t ypos = 56, const char* prefix = NULL, const char* suffix = NULL) {
  int16_t y = display.getCursorY();
  
  BGL_set(ON);

  display.setCursor(0, ypos);
  display.setTextColor(KS0108_OFF);

  for (int sz = 0; sz < 22; sz++)
    display.write(219);
  display.display();

  display.setTextColor(KS0108_ON);
  display.setCursor(0, ypos);

  if (prefix) display.print(prefix);
  display.print(s);
  if (suffix) display.print(suffix);
    
  display.display();
  display.setCursor(0, y);
}
/* ------------------------------------------------- */

int16_t GLCD_msgBox(const char* msg, bool preClear = true, int16_t divi = 2, int16_t padd = 20) {
  int16_t offs = display.width()/divi;
  int16_t x1, y1, x2, y2;
  int16_t N = (offs/divi)+padd;
  
  BGL_set(ON);

  if (preClear)
    display.clearDisplay();

  for(int16_t i=2; i<N; i++) {
    x1 = offs-i;
    y1 = (offs/divi)-(i/divi);
    x2 = i*divi;
    y2 = i;
    
    display.fillRect(x1, y1, x2, y2, KS0108_OFF);
    display.drawRect(x1, y1, x2, y2, KS0108_ON);
    display.display();
    delay(1);  
  }  

  for(int i=0; i<(N*2)-2; i+=4) {
    display.drawLine(x1, y1+10, x1+i, y1+10, 1);
    display.display(); // Update screen with each newly-drawn line
    delay(1);
  }

  display.setCursor(x1+40, y1+2);
  display.print("MSG");
  display.display();
  
  x1 = x1+4;
  y1 = y1+12;
  
  display.setTextWrap(false);
  display.setCursor(x1,y1);
  
  for (int i=0; i<strlen(msg); i++) {
    if ((display.getCursorX() > (N*2)+6) || (msg[i] == '\n' || msg[i] == '\r')) 
      display.setCursor(x1, y1=(y1+8));
    
    if (msg[i] != '\n' && msg[i] != '\r')
      display.write(msg[i]);
  }
  display.display();
  display.setTextWrap(true);

  return x1;
}
// ---------------------------------------------------------

void setBit(unsigned int &target, uint8_t x, bool set) {
  if (set)
    target |= (1 << x);
  else
    target &= ~(1 << x);

  target &= 0b0000111111111111;
}
// -------------------------------------------------- */

bool getBit(unsigned int target, uint8_t x) {
  return (bool)(target & (1 << x));
}
// -------------------------------------------------- */

void KP_t9_reset() {
  for (uint8_t x = 0; x < 4; x++)
    for (uint8_t y = 0; y < 3; y++)
      matrix[x][y].index = 0;
}
// ---------------------------------------------------------------

void clearBuf() {
  memset(buf_input, 0, sizeof(buf_input));
  buf_index = 0;
  KP_t9_reset();
  kp_last_press = millis();
}
// ---------------------------------------------------------------

uint8_t KP_t9_setMod(uint8_t index) {
  if (index < T9_MODES_N) {
    KP_t9_mode = KP_t9_modes[(t9_mod_mode=index)];
    return index;
  }
  return t9_mod_mode;
}
// ---------------------------------------------------------

void KP_showIcon(bool show = true) {
  BGL_set(ON);

  display.setCursor(120, 0);
  display.write(KP_t9_mode.icon);
  
  if (show)
    display.display(); 
}
// ---------------------------------------------------------

KeyMap *readKeyPad() {
  uint8_t col_pin, key = 0;
  bool    mod_key = false;
  
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      col_pin = matrix[row][col].pin;
        
      if (digitalRead(col_pin) == LOW) {
        if ((millis()-kp_last_press) > KP_DEBOUNCE) {
          key = matrix[row][col].key;
          kp_last_press = millis();
          
          while (digitalRead(col_pin) == LOW) {
            if ( ((millis()-kp_last_press) > 1000) && !mod_key) {
              if (key == 12 && (getState() == TXT_INPUT)) {
                KP_t9_setMod( (++t9_mod_mode %= T9_MODES_N) );
                KP_showIcon(true);
                mod_key = true;
              }
            }
          }
          digitalWrite(rowPins[row], HIGH);
          kp_last_press = millis();
          
          if (mod_key) 
            return nullptr;
          else
            return &(matrix[row][col]);
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  return nullptr;
}
// ---------------------------------------------------------------

bool KP_checkInput() {
  uint8_t key, index;
  char c;
  
  KeyMap *obj = readKeyPad();
  
  if (obj != NULL) {
    key = obj->key;

    if (obj->index < 1) {
      if (buf_input[buf_index] != '\0')
        buf_index = (key != 10 && key != 12) ? buf_index+1 : buf_index;
      KP_t9_reset();
    } 
    else if (key > 10)  
      buf_index++;
    
    if (key == 10) {
      buf_input[buf_index] = '\0';
      buf_index = (buf_index > 0) ? buf_index-1 : 0;
    }
      
    if (strlen(obj->t9_page) > 0) {
      index = (obj->index % strlen(obj->t9_page));
      c = obj->t9_page[index];

      if (t9_mod_mode > 0) {
        if (t9_mod_mode == T9_MODE_CAPS)
          c = (key != 11) ? c-32 : c;
      }
      buf_input[buf_index] = c;
      obj->index++;
    }
    return true;
  }
  return false;
}
// ---------------------------------------------------------------

void textInput() {
  setState(TXT_INPUT);
  ui_mode = UI_WRITE;
  clearBuf();
  KP_t9_mode = KP_t9_modes[T9_MODE_ALPHA];
  GLCD_print("> ", LCD_PRINT, true);
}
// ----------------------------------------

const char* addParams() {
  int16_t x1 = GLCD_msgBox("\n    Press '5'\n to Add Params.\n", false);
  int16_t i = x1;
  unsigned long t = millis();
  const char* value = nullptr;
  KeyMap *KP;
  
  BGL_set(ON);

  while ((millis()-t)<5000) {
    if (!((millis()-t)%(5000/92))) {      
      display.drawPixel(i++, 50, 1),
      display.display();
    }
    KP = readKeyPad();
    if ((KP != NULL) && (KP->key == 5))
      break;
  }
  display.clearDisplay();
  display.display();
  
  if ((KP != NULL) && (KP->key == 5)) {
    textInput();
    while (!BOOTSEL) {
      if (KP_checkInput()) { 
        display.clearDisplay();
        
        display.print("> ");
        display.println(buf_input);
        display.display();
      } 
    }
    ui_mode = UI_READ;
    setState(MENU_ON);
  }
  value = buf_input;
  return value;
}
// ---------------------------------------------------------

namespace tree {
  struct  TreeNode *node_active = nullptr;
  struct  TreeNode *node_select = nullptr;
  uint8_t level, sub_index;
  
  // ----------------------------------------
  
  uint8_t getId(uint16_t val = node_select->id, uint8_t lvl = level) {
    return ((val >> (lvl * 4)) & 0xF);
  }
  // -----------------------------------------
  
  void showTitle(const char* title, char fillChar = '-') {
    char screenBuf[22];
    int i, n = 0;
    int leftPad;
    int16_t xPos;
    
    memset(screenBuf, '\0', sizeof(screenBuf));

    for (i = 0; i < strlen(title); i++)
      if (isprint(title[i]))
        screenBuf[n++] = (isalpha(title[i])) ? toupper(title[i]) : title[i];
    
    leftPad = (((display.width()-1)/4) - n) / 2;

    BGL_set(ON);

    display.drawLine(0, 0, display.width(), 0, KS0108_ON);
    display.drawLine(0, 8, display.width(), 8, KS0108_ON);
    
    for (int l=2; l<8; l+=2)
      display.drawLine(0, l, (leftPad*4)-2, l, KS0108_ON);

    display.setFont(&Picopixel);
    display.setCursor((leftPad*4),6);    
    display.print(screenBuf);
    //display.display();
    xPos = display.getCursorX();
    
    for (int l=2; l<8; l+=2)
      display.drawLine(xPos, l, (display.width()-21), l, 1);

    display.drawLine((display.width()-21), 0, (display.width()-21), 8, 1);
    display.drawLine(display.width()-1,0,display.width()-1,8,1);
    display.drawLine(0, 0, 0, 8, 1);
    
    display.setFont(NULL);
    //display.setCursor(0,10);
    display.display();
  }
  // ----------------------------------------

  
  void Render(struct TreeNode* node = node_active) {
    struct TreeNode* child = NULL;
  
    if (node && node->sub_count) {
      BGL_set(ON);
      display.clearDisplay();
      
      showTitle(node_active->label, 176);
      display.setCursor(2,14);

      for (uint8_t i = 0; i < node->sub_count; i++) {
        display.setTextColor(KS0108_ON);
        child = node->children[i];
  
        if (!child) continue;
        if (node_select->id == child->id)
          display.setTextColor(KS0108_OFF, KS0108_ON);

        display.print( (child->sub_count) ? "/ " : ((child->action != NULL) ? "* " : "? "));
        display.println(child->label);
        display.display();
  
        display.setTextColor(KS0108_ON);
      }
    }
  }
  // ----------------------------------------
  
  void Select(uint8_t index, struct TreeNode* node = node_active) {
    node_active = node;
    node_select = (node->sub_count == 0) ? node : node->children[0];
    sub_index = (node->sub_count == 0) ? 1 : sub_index;
  
    if (node->sub_count == 0)
      return;
  
    index = ((index > node->sub_count) ? 1 : ((index < 1) ? node->sub_count : index));
    node_select = node->children[index - 1];
    sub_index = index;
  
    Render(node);
  }
  // ----------------------------------------
  
  bool Init(struct TreeNode* node = &(menuTree), bool initialized = true) {
    if (!node)
      return initialized;
  
    if (!initialized) {
      if (node-> id != 0x0001)
        node = &(menuTree);
  
      level = sub_index = 1;
      initialized = true;
  
      Select(1, node);
    }
  
    if (node->sub_count) {
      for (uint8_t i = 0; i < node->sub_count; i++) {
        (*node->children[i]).parent = node;
        Init( node->children[i], initialized);
      }
    }
  
    return initialized;
  }
  // ----------------------------------------

  void GoTo(struct TreeNode* node) {
    if (!node)
      return;
  
    if (node->id < node_active->id) {
      Select(1, node);
      level = (level > 1) ? level - 1 : 1;
    }
    else {
      if (node->sub_count) {
        Select(1, node);
        level++;
      }
      else {
        if (node->action != NULL) {
          const char *p = (node->params == NULL) ? addParams() : node->params;
          node->action(p);
        }
        else
          GLCD_overwrite("void..");
      }
    }
  }
  // ----------------------------------------
  
  void Navigate(uint8_t key) {
    if (key == BTN_UP) {
      sub_index--;
      Select(sub_index);
    }
    else if (key == BTN_DOWN) {
      sub_index++;
      Select(sub_index);
    }
    else if (key == BTN_ENTER) {
      GoTo(node_select);
    }
    else if (key == BTN_BACK) {
      if (node_active-> id == 0x0001)
        Home();
      else
        GoTo(node_active->parent);
    }
  }
}
// ---------------------------------------------------------

void Reader(const char* txt_buf = nullptr) {
  //const char  *txt_buf;
  uint16_t    last, index;
  uint8_t     key;
  bool        valid, curstate;
  KeyMap      *KP;

  if (txt_buf != NULL && strlen(txt_buf)) {
    setState(READER);
    
    while (1) {
      curstate = false;
      index    = 0;
      
      BGL_set(ON);
      display.clearDisplay();
      display.display();
      delay(100);
      
      while (index < strlen(txt_buf)-1) {
        BGL_set(ON);
        display.write(txt_buf[index++]);
        display.display();
        delay(10);
        
        if (display.getCursorY() > 56) {
          last  = index;
          valid = false;
          key   = 0;
          
          while (!valid) {
            KP = readKeyPad();
            
            if (KP != NULL) {
              key = KP->key;
              
              if (key == 7 || key == 10 || key == 12) {
                valid = false;
                break;
              }
              else
                valid = true;
            }
            if ( !(millis() % 200) ) 
              GLCD_blinkCursor((curstate = !curstate), (20*6), 56, 25);
          }
          GLCD_blinkCursor(false, (20*6), 56, 25);
          
          if (KP != NULL) {
            if (!valid) return;
            
            display.pushUp(8);
            display.setCursor(0, 56);
            display.display();
          }
        }
      }
      GLCD_blinkCursor(true, (20*6), 56, 19);
      
      while ((readKeyPad() == NULL) && !BOOTSEL);
      GLCD_msgBox("\n  READ AGAIN?\n\n    PRESS 5", false);
      
      while (1) {
        KP = readKeyPad();
        if (KP != NULL) break;
      }
      if (KP->key != 5) return;
    } 
    setState(sys_status_prev.id);   
  }
}
// ---------------------------------------------------------

void printFuncName(const char* id, uint8_t ypos = 56) {
  GLCD_overwrite(id, ypos, "'", "()'");
}

// ---------------------------------------------------------

void Menu() {
  ui_mode = UI_READ;
  setState(MENU_ON);
  
  if (!menu_init) 
    menu_init = tree::Init(&menuTree, menu_init);

  tree::Render();
}
// ---------------------------------------------------------

void awaitResponse() {
  setState(AWAIT_RESP);
  ui_mode = UI_READ;
  await_response = true;
  resp_timer = millis();
  GLCD_anim();
}
// ---------------------------------------------------------


void sendCommand(bool confirm = true, uint8_t cmd_type = cmd_type_override) {
  KeyMap *KP;
  
  if (confirm) {
    GLCD_msgBox("\n  SEND COMMAND?\n\n    PRESS 5", false);

    while (1) {
      KP = readKeyPad();
      if (KP != NULL) break;
    }
  }
  
  if ((KP->key == 5) || !confirm) {
    display.clearDisplay(); 
    display.display();
    delay(1000);

    if (cmd_type == CMD_HID) 
      Keyboard.println(buf_input);
    else 
      Serial.println(buf_input);
    
    awaitResponse();
  }
  else // return to menu
    Menu(); 
}
// ---------------------------------------------------------

void sketcher() {
  unsigned long t = millis();
  int16_t thisX   = display.width()/2;
  int16_t thisY   = display.height()/2;
  bool last_pixel = false;
  bool active     = true;
  bool printOut   = false;
  KeyMap *KP;
  
  BGL_set(ON);

  display.clearDisplay();
  display.drawRect(0, 0, display.width(), display.height(), 1);
  display.display();
    
  while (active) {
    KP = readKeyPad();
    
    if (KP != NULL) {
      uint8_t key = KP->key;

      BGL_set(ON);
      display.drawPixel(thisX, thisY, 0);

      if (last_pixel)
        display.drawPixel(thisX, thisY, 1);
                
      if (key == 2 || key == 4 || key == 6 || key == 8) {  
        if (key == 2)
          thisY = (thisY > 0) ? thisY-1 : thisY;
        if (key == 8)
          thisY = (thisY < display.height()) ? thisY+1 : thisY;
        if (key == 4)
          thisX = (thisX > 0) ? thisX-1 : thisX;
        if (key == 6)
          thisX = (thisX < display.width()) ? thisX+1 : thisX;
      }
      else if (key == 5) {
        display.drawPixel(thisX, thisY, KS0108_INVERSE);
      }
      last_pixel = display.getPixel(thisX, thisY);
      display.display();   
    }
    else if (KP == NULL) {
      if ( !(millis()%200) )
        display.drawPixel(thisX, thisY, KS0108_INVERSE),
        display.display();

      if (BOOTSEL) {
        t = millis();
        BGL_set(ON);

        while (BOOTSEL) {
          uint32_t elapsed = ((millis()-t)/10);
          KP = readKeyPad();
          
          if (KP != NULL && (KP->key > 9)) {
            active = false;
            break;
          }
          if (elapsed <= 384) {  
            if (elapsed<128)
              display.drawLine(0, 0, elapsed, 0, 0);
            else if (elapsed >= 128 && elapsed < 192)
              display.drawLine(127, 0, 127, (elapsed-128), 0);
            else if (elapsed >= 192 && elapsed < 320)
              display.drawLine(127, 63, 127-(elapsed-192), 63, 0);
            else if (elapsed >= 320 && elapsed < 384) 
              display.drawLine(0, 63, 0, 63-(elapsed-320), 0);
            display.display();
            delay(4);
          }
          else if (elapsed > 384) {
            printOut = true;
            active = false;
          }
        }
        display.drawRect(0, 0, display.width(), display.height(), 1);
        display.display();       
      }
    }
  }
  
  if (printOut) {
    delay(1000);
    display.dumpPBM();
    delay(1000);
    t = millis();
    GLCD_msgBox("\n    SUCCESS!\n BUFFER PRINTED", false);
    while ((millis()-t)<3000) {
      KP = readKeyPad();
      if (KP != NULL)
        break;
    }
  }
}
// ---------------------------------------------------------

String sys_dateTimeStr(const int (&arr)[3], const char* sep = ":") {
  String val = "";
  int index = 0;

  for (const int &n : arr) 
    val = val
    + ((n<10) ? "0" : "") 
    + String(n) 
    + ((++index < 3) ? sep : "");

  return val;
}
// ---------------------------------------------------------

void cmd_sendLcdBuf(const char* param = nullptr) {
  printFuncName(__func__);
  delay(500);
  BGL_set(ON);

  display.dumpPBM();

  display.setCursor(128-(5*6), 56);
  display.print(" DONE");
  display.display();
}
// ---------------------------------------------------------

void cmd_myFun(const char* param = nullptr) {
  printFuncName(__func__);
  delay(500);
  BGL_set(ON);

  for (int i = 0; i < 5; i++)
    digitalWrite(28, !digitalRead(28)),
    delay(500);
  digitalWrite(28, HIGH);

  display.setCursor(128-(5*6), 56);
  display.print(" DONE");
  display.display();
}
// ---------------------------------------------------------

void cmd_txtInput(const char* param = nullptr) {
  printFuncName(__func__);
  delay(500);
  BGL_set(ON);
  
  if (param != NULL) 
    cmd_type_override = (strcmp(param,"ser") == 0) ? CMD_SER : CMD_HID;
  textInput();
  cmd_type_override = CMD_HID;
}
// ---------------------------------------------------------

void cmd_sketch(const char* param = nullptr) {
  printFuncName(__func__);
  delay(500);
  BGL_set(ON);
  
  sketcher();
  Menu();
}

void clock_set(const char* raw) {
  unsigned long x = strtol(raw, 0, 10);
  setTime(x);
}
// ---------------------------------------------------------
