#pragma once

#include <pico/stdlib.h>
//#include <hardware/pwm.h>

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#define ArrLen(x)         (sizeof(x)/sizeof(x[0]))
#define bit2byte(x,pos)   (((x) >> ((pos) * 4)) & 0xF)

#define LED_PIN           LED_BUILTIN
#define MCU_CLOCK         (133000000.0)
#define RESP_TIMEOUT      30000UL
#define BGL_TIMEOUT       60000UL
#define SERIAL_TIMEOUT    2000UL
#define READ_DEL          250
#define LCD_MAX_CHARS     168
#define SERIAL_BUF_SIZE   255
#define HEADER_SIZE       64
#define BGL_PIN           28
#define KP_DEBOUNCE       30
#define TIME_REQUEST      7
#define DEVICE_VID        0x0920
#define DEVICE_PID        0x1986
#define MANUFACTURER      "data_dogenigt"
#define DEVICE_NAME       "LoPi-Phone v1.0"

// _______________________________________________________________________

const char loremipsum[] = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.";

constexpr uint8_t row_N   = 4;
constexpr uint8_t col_N   = 6;
constexpr int16_t curSym  = 219;
constexpr char ae = -26; //   xE6 \u00E6
constexpr char oe = -8;  //   xF8 \u00F8
constexpr char aa = -27; //   xE5 \u00E5

KS0108_GLCD display = KS0108_GLCD(
  22, // -> LCD 4  ( RS )
  20, // -> LCD 6  ( E )
  21, // -> LCD 7  ( D0 )
  19, // -> LCD 8  ( D1 )
  18, // -> LCD 8  ( D2 )
  17, // -> LCD 10 ( D3 )
  16, // -> LCD 11 ( D4 )
  27, // -> LCD 12 ( D5 )
  26, // -> LCD 13 ( D6 )
  5,  // -> LCD 14 ( D7 )
  4,  // -> LCD 15 ( CS1 )
  3,  // -> LCD 16 ( CS2 )
  2   // -> LCD 17 ( RES )
);

uint8_t rowPins[row_N] = {
  14, // -> KP 2   ( R1 )
  11, // -> KP 5   ( R2 )
  9,  // -> KP 7   ( R3 )
  8   // -> KP 8   ( R4 )
};

uint8_t colPins[col_N] = {
  15, // -> KP 1   ( C1 )
  13, // -> KP 3   ( C2 )
  12, // -> KP 4   ( C3 )
  10, // -> KP 6   ( C4 )
  7,  // -> KP 9   ( C5 )
  6   // -> KP 10  ( C6 )
};

// - - - - - - - - - - - -

enum : uint8_t {
  BTN_UP    = 2,
  BTN_LEFT  = 4,
  BTN_RIGHT = 6,
  BTN_DOWN  = 8,
  BTN_ENTER = 5,
  BTN_BACK  = 10
};

enum : uint8_t {
  LCD_PRINT,
  LCD_PRINTLN,
  LCD_WRITE
};

enum : uint8_t {
  UI_READ,
  UI_WRITE,
  UI_STDIN,
  UI_STDIN_DONE
};

enum : uint8_t {
  T9_MODE_ALPHA,
  T9_MODE_CAPS,
  T9_MODE_NUM,
  T9_MODES_N
};

enum : uint8_t { 
  CMD_HID, 
  CMD_SER 
};

enum : bool { 
  OFF = false,
  ON  = true
};

enum : int { 
  KP_MOD = -1 
};

typedef enum : int8_t {
  OTHER = -1,
  BOOTING,
  BOOTED,
  HOME_ON,
  MENU_ON,
  STDIN_MSC,
  STDIN_SER,
  READER,
  TXT_INPUT,
  AWAIT_RESP
} state_enum;

// - - - - - - - - - - - -

char nordic_chars[4] = {0xE6, 0xF8, 0xE5, 0x00};

struct KeyMap {
  uint8_t     pin;
  uint8_t     key;
  int16_t     index;
  const char* t9_page;
};
KeyMap matrix[4][3] = {
  {
    { colPins[0], 1, 0, nordic_chars},
    { colPins[2], 2, 0, "abc"},
    { colPins[1], 3, 0, "def"},
  },
  {
    { colPins[0], 4, 0, "ghi"},
    { colPins[2], 5, 0, "jkl"},
    { colPins[3], 6, 0, "mno"}
  },
  {
    { colPins[0], 7, 0, "pqrs"},
    { colPins[2], 8, 0, "tuv"},
    { colPins[3], 9, 0, "wxyz"}
  },
  {
    { colPins[2], 10, 0, ""},
    { colPins[5], 11, 0, " .,!?:;-_/()<>+=*'\"%$&#@{}[]"},
    { colPins[4], 12, 0, ""}
  },
};

// - - - - - - - - - - - -

struct _systatus {
  state_enum  id;
  const char *txt;
};
_systatus sys_states[] = {
  {BOOTING,   "Booting."},
  {BOOTED,    "System ready."},
  {HOME_ON,   "Home screen."},
  {MENU_ON,   "Menu launched."},
  {STDIN_MSC, "Reading MSC in."},
  {STDIN_SER, "Reading Serial in."},
  {READER,    "File reader."},
  {TXT_INPUT, "Text input."},
  {AWAIT_RESP, "Awaiting response."}
};
_systatus
sys_status_prev,
sys_status = sys_states[BOOTING];

// - - - - - - - - - - - -

struct t9_mode_t {
  const char* label;
  uint8_t     enum_t;
  int16_t     icon;
}
KP_t9_modes[3] = {
  {"alpha_std",  T9_MODE_ALPHA, 42},
  {"alpha_caps", T9_MODE_CAPS,  94},
  {"numeric",    T9_MODE_NUM,   241},
};

t9_mode_t      KP_t9_mode;
uint8_t        t9_mod_mode;

struct _sysData {
  char name[8];
  char content[16];
};

_sysData homeScreen[7] = {
  {"date",""},
  {"time",""},
  {"sun", ""},
  {"day", ""},
  {"moon",""},
  {"temp",""},
  {"ip",  ""},
};

_sysData sysData[1] = {
  {"unread",""},
};

/* ------------------------------------------------- */

const char* months[12] = {
  "Jan","Feb","Mar",
  "Apr","May","Jun",
  "Jul","Aug","Sep",
  "Oct","Nov","Dec"
};

/* ------------------------------------------------- */
 
char           buf_input[LCD_MAX_CHARS*8]; // (21*8 = 168) * 8 = 1344
char           buf[256];
char           dt_str[20];
char           hex_str[2]; 
uint8_t        buf_index;
uint8_t        ui_mode;
uint8_t        bgl_pwm;
uint8_t        cmd_type_override;
int16_t        clock_cur_pos;
unsigned long  moon_day;
unsigned long  bgl_timeout, bgl_timer;
unsigned long  serial_timer, resp_timer;
unsigned long  kp_last_press, T;
unsigned long  screenUpdate, clock_update;
unsigned int   btn_map;
bool           bgl_state;
bool           bootsel_held;
bool           menu_init;
bool           await_response;
bool           data_stored, data_updated;
bool           timeSynced;

/* ------------------------------------------------- */

// Function declarations
void Home(bool init = false);
void Menu();
void sysTick();
void clock_store();
void clock_tick(bool update = false);
void clock_show(const char* str = dt_str, int16_t &cur = clock_cur_pos);
void printTopBar(const char* title = nullptr);
void printHomeItems(int16_t xPos, int16_t yPos);
void GLCD_indent(uint8_t margin, bool disp = false, char c = ' ');
bool KP_T9_getChar(KeyMap *obj);

// ___________________________________________________

void setMoonDay(const char* raw) {
  unsigned long x = strtol(raw, 0, 10);
  moon_day = x;
}
// ---------------------------------------------------------

void setState(int8_t state, const char* custom_txt = NULL) {
  sys_status_prev = sys_status;
  sys_status = sys_states[state];
}
// ----------------------------------------

state_enum getState(_systatus* state = &sys_status) {
  return state->id;
}
// ----------------------------------------

char* hex2char(unsigned char x, char hex[] = hex_str) {
  hex[0] = (char)x;
  hex[1] = '\0';
  return hex;
}
// -------------------------------------------------

int16_t ascii2lcd(int16_t input) {
  switch(input) {
     case 230: return 145;  //  æ
     case 248: return 236;  //  ø
     case 229: return 134;  //  å
     case 198: return 146;  //  Æ
     case 216: return 237;  //  Ø
     case 197: return 143;  //  Å
     default: return input; // (*)
  }
}
// -------------------------------------------------

bool bootHeld(uint32_t limit = 2000) {
  uint32_t timer = millis();
  bool states[2] = {true, true};
  bool held = true;

  while (held) {
    states[0] = (BOOTSEL) ? true : false;
    
    if (states[0] != states[1]) {
      if (((millis()-timer) > limit))
        return states[0];
      timer = millis();
      states[1] = states[0];
    }
  }
  return false;
}
// -------------------------------------------------

char int2char(int n) {
  return (n > 9) ? ' ' : n+'0';
}
// -------------------------------------------------

int char2int(char c) {
  return (isdigit(c)) ? int(c-'0') : -1;
}
// -------------------------------------------------

int str2int(char c) {
  return (isdigit(c)) ? int(c-'0') : -1;
}

// -------------------------------------------------

void BGL_set(bool state, unsigned long dur = BGL_TIMEOUT) {
  if (state) {
    analogWrite(BGL_PIN, bgl_pwm = 255);
    bgl_timer = millis();
    bgl_timeout = dur;
  }
  else {
    if (bgl_pwm > 0) {
      if ( (millis()-bgl_timer)%200 == 0) {
        bgl_pwm--;
        analogWrite(BGL_PIN, bgl_pwm);
      }
    }
  }
}
// ---------------------------------------------------------

void GLCD_decodeStr(const char* str) {
  for (int i=0; i<strlen(str); i++) {
    int hx = (unsigned char)str[i];
    if (hx==197||hx==198||hx==216 ||
        hx==229||hx==230||hx==248)
        hx = ascii2lcd(hx);
    else
        hx = (int)str[i];

    if (display.getCursorX() >= 120)
        display.println();
    display.write(hx);
  }
}
// ---------------------------------------------------------

void GLCD_blinkCursor(bool state, int16_t x=0, int16_t y=56, int16_t c=219) {

  display.setCursor(x,y);
  display.setTextColor((state)? KS0108_ON : KS0108_OFF);
  display.write(c);
  display.display();
  display.setTextColor(KS0108_ON);
}
// ---------------------------------------------------------

void GLCD_overflow(int maxY = 56, int height = 8) {
  if (display.getCursorY() > maxY)
    display.pushUp(height),
    display.setCursor(0, maxY);
}
// ---------------------------------------------------------------

void GLCD_anim() {
  int16_t i, ripl = 2;
  unsigned long t = millis();

  display.clearDisplay();
  
  for (i = 0; i < max(display.width(), display.height())/2; i += 2) {
    if ((Serial.available() || stdin_available)) break;
    display.drawBitmap((display.width()-BMP_WIDTH)/2, (display.height()-BMP_HEIGHT)/2, cat_bmp, BMP_WIDTH, BMP_HEIGHT, 1); 
    
    if (i >= ripl)
      display.drawCircle( display.width()/2, display.height()/2, (i-ripl), KS0108_OFF );
    
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_OFF );
    display.display();
    delay(30);
  }

  for (; i < (max(display.width(), display.height())/2)+(10 +ripl); i += 2) {
    if ((Serial.available() || stdin_available)) break;
    display.drawBitmap((display.width()-BMP_WIDTH)/2, (display.height()-BMP_HEIGHT)/2, cat_bmp, BMP_WIDTH, BMP_HEIGHT, 1);  
    display.drawCircle( display.width()/2, display.height()/2, (i-ripl), KS0108_OFF );
    display.drawCircle( display.width()/2, display.height()/2, i, KS0108_OFF );
    display.display();
    delay(30);
  }   
}
// ---------------------------------------------------------------

bool GLCD_Init() {
  BGL_set(ON, 0);

  if ( display.begin(KS0108_CS_ACTIVE_HIGH) == false )
    return false;

  display.display();                // display flash
  delay(3000);

  display.setFont(NULL);
  display.clearDisplay();           // Clear the buffer
  display.setTextSize(1);           // Normal 1:1 pixel scale
  display.setTextColor(KS0108_ON);  // Draw white text
  display.setCursor(0, 0);          // Start at top-left corner
  display.cp437(true);              // Use full 256 char 'Code Page 437' font
  display.setTextWrap(true);        // Toggle text wrap (else clip right).
  display.display();

  return true;
}
/* ------------------------------------------------- */

void GLCD_print(const char* txt, uint8_t mode = LCD_PRINT, bool preClear = false) {
  if (preClear)
    display.clearDisplay();

  display.print(txt);

  if (mode == LCD_PRINTLN)
    display.print("\n");

  display.display();
}
/* ------------------------------------------------- */

template <typename T>
void GLCD_print(T txt, uint8_t mode, bool preClear) {
  //BGL_SET(ON);

  if (preClear)
    display.clearDisplay(),
    display.setCursor(0, 0);

  switch (mode) {
    default:
    case 0:
      display.print(txt);
      break;
    case 1:
      display.println(txt);
      break;
    case 2:
      display.print(txt);
      break;
  }
  display.display();
}

/* ------------------------------------------------- */

void GLCD_indent(uint8_t margin, bool disp, char c) {
  if (margin == 0) return;

  //BGL_SET(ON);

  for (uint8_t n = 0; n < margin; n++)
    display.write(c);

  if (disp)
    display.display();
}
// ---------------------------------------------------------

void GLCD_overwrite(const char* s = " ", int16_t ypos = 56, const char* prefix = NULL, const char* suffix = NULL) {
  int16_t y = display.getCursorY();
  
  //BGL_SET(ON);

  display.setCursor(0, ypos);
  display.setTextColor(KS0108_OFF);

  for (int sz = 0; sz < 22; sz++)
    display.write(219);
  display.display();

  display.setTextColor(KS0108_ON);
  display.setCursor(0, ypos);

  if (prefix) display.print(prefix);
  display.print(s);
  if (suffix) display.print(suffix);
    
  display.display();
  display.setCursor(0, y);
}
/* ------------------------------------------------- */

int16_t GLCD_msgBox(const char* msg, bool preClear = true, int16_t divi = 2, int16_t padd = 20) {
  int16_t offs = display.width()/divi;
  int16_t x1, y1, x2, y2;
  int16_t N = (offs/divi)+padd;
  
  BGL_set(ON);

  display.setFont(NULL);
  display.setTextWrap(true);
  display.setTextSize(1);

  if (preClear)
    display.clearDisplay();

  for(int16_t i=2; i<N; i++) {
    x1 = offs-i;
    y1 = (offs/divi)-(i/divi);
    x2 = i*divi;
    y2 = i;
    
    display.fillRect(x1, y1, x2, y2, KS0108_OFF);
    display.drawRect(x1, y1, x2, y2, KS0108_ON);
    display.display();
    delay(1);  
  }  

  for(int i=0; i<(N*2)-2; i+=4) {
    display.drawLine(x1, y1+10, x1+i, y1+10, 1);
    display.display();
    delay(1);
  }

  display.setCursor(x1+40, y1+2);
  display.print("MSG");
  display.display();
  
  x1 = x1+4;
  y1 = y1+12;
  
  display.setTextWrap(false);
  display.setCursor(x1,y1);
  
  for (int i=0; i<strlen(msg); i++) {
    if ((display.getCursorX() > (N*2)+6) || (msg[i] == '\n' || msg[i] == '\r')) 
      display.setCursor(x1, y1=(y1+8));
    
    if (msg[i] != '\n' && msg[i] != '\r')
      display.write(msg[i]);
  }
  display.display();
  display.setTextWrap(true);

  return x1;
}
// ---------------------------------------------------------

void setBit(unsigned int &target, uint8_t x, bool set) {
  if (set)
    target |= (1 << x);
  else
    target &= ~(1 << x);

  target &= 0b0000111111111111;
}
// -------------------------------------------------- */

bool getBit(unsigned int target, uint8_t x) {
  return (bool)(target & (1 << x));
}
// -------------------------------------------------- */

void KP_t9_reset() {
  for (uint8_t x = 0; x < 4; x++)
    for (uint8_t y = 0; y < 3; y++)
      matrix[x][y].index = 0;
}
// ---------------------------------------------------------------

void clearBuf() {
  memset(buf_input, 0, sizeof(buf_input));
  buf_index = 0;
  KP_t9_reset();
  kp_last_press = millis();
}
// ---------------------------------------------------------------

void Serial_flush(int timeout = 1000) {
  unsigned long t = millis();

  while ((millis()-t)<timeout) {
    if (Serial.available() > 0) {
      char c = Serial.read();
    }
  }
}
// ---------------------------------------------------------------

void KP_showIcon(bool show = true, bool clear = false) {
  display.setCursor(120, 0);
  display.fillRect(120, 0, 7, 7, 0);
  display.setTextColor((clear)?0:1);
  display.write(KP_t9_mode.icon);
  display.setTextColor(1);

  if (show)
    display.display(); 
}
// ---------------------------------------------------------

uint8_t KP_t9_setMod(uint8_t index) {
  if (index < T9_MODES_N) {
    KP_t9_mode = KP_t9_modes[(t9_mod_mode=index)];
    KP_t9_reset();
    return index;
  }
  return t9_mod_mode;
}
// ---------------------------------------------------------

KeyMap *readKeyPad() {
  uint8_t col_pin, key = 0;
  bool    num_key, mod_key = false;
  
  for (uint8_t row = 0; row < 4; row++) {
    digitalWrite(rowPins[row], LOW);
    
    for (uint8_t col = 0; col < 3; col++) {
      col_pin = matrix[row][col].pin;
        
      if (digitalRead(col_pin) == LOW) {
        if ((millis()-kp_last_press) > KP_DEBOUNCE) {
          key = matrix[row][col].key;
          kp_last_press = millis();
          
          uint8_t mod_stored = t9_mod_mode;
          
          while (digitalRead(col_pin) == LOW) {
            if ((millis()-kp_last_press) > 1000) {
              key = (key == 11) ? 0 : key;
              if (key < 10) {
                KP_t9_setMod(T9_MODE_NUM);
                if (mod_stored != T9_MODE_NUM)
                  KP_showIcon(true),
                  mod_stored = T9_MODE_NUM;
              }
              else if ((key == 12) && !mod_key) {
                KP_t9_setMod(!t9_mod_mode);
                if (!mod_key)
                  KP_showIcon(true);
                mod_key = true;
              }
            }
          }
          digitalWrite(rowPins[row], HIGH);
          
          if (mod_key) 
            return nullptr;
          else
            return &(matrix[row][col]);
        }
      }
    }
    digitalWrite(rowPins[row], HIGH);
  }
  return nullptr;
}
// ---------------------------------------------------------------

uint8_t KP_t9_dictLen(int16_t *arr) {
  int i = 0;
  for (; i < ArrLen(matrix[0][0].t9_page); i++)
    if (arr[i] == 0)
      break;
  return i;
}
// ---------------------------------------------------------------

bool KP_T9_getChar(KeyMap *obj) {
  if (obj == NULL) 
    return false;

  int16_t index;
  uint8_t key = obj->key;
  char c;

  key = (key == 11) ? 0 : key;

  if (obj->index < 1) {
    if (buf_input[buf_index] != '\0')
      buf_index = (key != 10 && key != 12) ? buf_index + 1 : buf_index;
    KP_t9_reset();
  } 
  else {
    if (key == 12)  
      buf_index++;
  }
  
  if (key == 10) {
    buf_input[buf_index] = '\0';
    buf_index = (buf_index > 0) ? buf_index - 1 : 0;
  }
    
  if (t9_mod_mode == T9_MODE_NUM) {
    if (key != 10 && key != 12) {
      int16_t x = (key+48);
      buf_input[buf_index] = x;
      obj->index++;
    }
    KP_t9_setMod(T9_MODE_ALPHA);
    KP_showIcon(true);
  }
  else if (strlen(obj->t9_page) > 0) {
    index = obj->index % strlen(obj->t9_page);
    c = obj->t9_page[index];

    if (t9_mod_mode == T9_MODE_CAPS)
      c = (key != 0) ? c-32 : c;

    buf_input[buf_index] = c;
    obj->index++;
  }
  return true;
}

// ---------------------------------------------------------------

int numInput(size_t num_size = 4, const char* label = "> ") {
  char num_buf[num_size+1] = {'\0'};
  int n, index = 0;

  GLCD_print(label, 0, 1);

  while (1) {
    KeyMap *KP = readKeyPad();
    if (KP != NULL) {
      if (KP->key == 12) break;
      n = (KP->key < 10) ? KP->key : ((KP->key == 11) ? 0 : -1);

      if (n < 0) {
        index = (index > 0) ? index-1 : 0;
        num_buf[index] = '\0';
      }
      else {
        if (index < num_size)
          num_buf[index++] = n+48;
      }
      GLCD_print(label, 0, 1);
      GLCD_print(num_buf, 0, 0);
    }
    else if (BOOTSEL) {
      while (BOOTSEL) yield();
      break;
    }
  }
  return (strlen(num_buf)) ? atoi(num_buf) : -1;
}

// ---------------------------------------------------------------

void textInput(const char* label = "> ") {
  KP_t9_setMod(T9_MODE_ALPHA);

  setState(TXT_INPUT);
  clearBuf();
  GLCD_print(label, LCD_PRINT, true);
  display.write(curSym);

  KP_showIcon(true);
}
// ----------------------------------------

const char* addParams(const char* text = "\n    Press '5'\n to Add Params.\n") {
  int16_t x1        = GLCD_msgBox(text, false);
  int16_t i         = x1;
  unsigned long t   = millis();
  const char* value = nullptr;
  KeyMap *KP;
  
  while ((millis()-t)<5000) {
    if (!((millis()-t)%(5000/110))) {      
      display.drawPixel(i++, 50, 1),
      display.display();
    }
    KP = readKeyPad();
    if ((KP != NULL) && (KP->key == 5))
      break;
  }
  display.clearDisplay();
  display.display();
  
  if ((KP != NULL) && (KP->key == 5)) {
    textInput();
    while (!BOOTSEL) {
      KP = readKeyPad();
      if (KP_T9_getChar(KP)) { 
        GLCD_print("> ",0,1);
        GLCD_decodeStr(buf_input);
        display.write(curSym);
        KP_showIcon(true);
      } 
    }
  }
  value = buf_input;
  return value;
}
// ---------------------------------------------------------

namespace tree {
  struct  TreeNode *node_active = nullptr;
  struct  TreeNode *node_select = nullptr;
  uint8_t level, sub_index;
  
  // ----------------------------------------
  
  uint8_t getId(uint16_t val = node_select->id, uint8_t lvl = level) {
    return ((val >> (lvl * 4)) & 0xF);
  }
  // -----------------------------------------
  
  void showTitle(const char* title, char fillChar = '-') {
    char screenBuf[22];
    int i, n = 0;
    int leftPad;
    int16_t xPos;
    
    memset(screenBuf, '\0', sizeof(screenBuf));

    for (i = 0; i < strlen(title); i++)
      if (isprint(title[i]))
        screenBuf[n++] = (isalpha(title[i])) ? toupper(title[i]) : title[i];
    
    leftPad = (((display.width()-1)/4) - n) / 2;

    display.drawLine(0, 0, display.width(), 0, KS0108_ON);
    display.drawLine(0, 8, display.width(), 8, KS0108_ON);
    
    for (int l=2; l<8; l+=2)
      display.drawLine(0, l, (leftPad*4)-2, l, KS0108_ON);

    display.setFont(&Picopixel);
    display.setCursor((leftPad*4),6);    
    display.print(screenBuf);

    xPos = display.getCursorX();
    
    for (int l=2; l<8; l+=2)
      display.drawLine(xPos, l, (display.width()-21), l, 1);

    display.drawLine((display.width()-21), 0, (display.width()-21), 8, 1);
    display.drawLine(display.width()-1,0,display.width()-1,8,1);
    display.drawLine(0, 0, 0, 8, 1);
    
    display.setFont(NULL);
    //display.setCursor(0,10);
    display.display();
  }
  // ----------------------------------------

  
  void Render(struct TreeNode* node = node_active) {
    struct TreeNode* child = NULL;
  
    if (node && node->sub_count) {
      display.clearDisplay();
      
      showTitle(node_active->label, 176);
      display.setCursor(0,14);

      for (uint8_t i = 0; i < node->sub_count; i++) {
        display.setTextColor(KS0108_ON);
        child = node->children[i];
  
        if (!child) continue;
        if (node_select->id == child->id)
          display.setTextColor(KS0108_OFF, KS0108_ON);

        display.print( (child->sub_count) ? "/ " : ((child->action != NULL) ? "* " : "? "));
        display.println(child->label);
        display.display();
        display.setTextColor(KS0108_ON);
      }
      display.setTextColor(KS0108_ON);
      printTopBar();
    }
  }
  // ----------------------------------------
  
  void Select(uint8_t index, struct TreeNode* node = node_active, bool show = true) {
    node_active = node;
    node_select = (node->sub_count == 0) ? node : node->children[0];
    sub_index = (node->sub_count == 0) ? 1 : sub_index;
  
    if (node->sub_count == 0)
      return;
  
    index = ((index > node->sub_count) ? 1 : ((index < 1) ? node->sub_count : index));
    node_select = node->children[index - 1];
    sub_index = index;
  
    if (show)
      Render(node);
  }
  // ----------------------------------------
  
  bool Init(struct TreeNode* node = &(menuTree), bool initialized = true) {
    if (!node)
      return initialized;
  
    if (!initialized) {
      if (node-> id != 0x0001)
        node = &(menuTree);
  
      level = sub_index = 1;
      initialized = true;
  
      Select(1, node, false);
    }
  
    if (node->sub_count) {
      for (uint8_t i = 0; i < node->sub_count; i++) {
        (*node->children[i]).parent = node;
        Init( node->children[i], initialized);
      }
    }
  
    return initialized;
  }
  // ----------------------------------------

  void GoTo(struct TreeNode* node) {
    if (!node)
      return;
  
    if (node->id < node_active->id) {
      Select(1, node);
      level = (level > 1) ? level - 1 : 1;
    }
    else {
      if (node->sub_count) {
        Select(1, node);
        level++;
      }
      else {
        if (node->action != NULL) {
          const char *p = (node->params == NULL) ? addParams() : node->params;
          node->action(p);
        }
        else
          GLCD_overwrite("void..");
      }
    }
  }
  // ----------------------------------------
  
  void Navigate(uint8_t key) {
    BGL_set(ON);
    
    if (key == BTN_UP) {
      sub_index--;
      Select(sub_index);
    }
    else if (key == BTN_DOWN) {
      sub_index++;
      Select(sub_index);
    }
    else if (key == BTN_ENTER) {
      GoTo(node_select);
    }
    else if (key == BTN_BACK) {
      if (node_active-> id == 0x0001)
        Home();
      else
        GoTo(node_active->parent);
    }
  }
}
// ---------------------------------------------------------

void showWarning(const char* msg, unsigned long timeout = 5000, bool preClear = false) {
  unsigned long t = millis();

  int16_t i = GLCD_msgBox(msg, preClear);

  while ((millis()-t)<timeout) {
    if (!((millis()-t)%(timeout/110))) {      
      display.drawPixel(i++, 50, 1),
      display.display();
    }
    if (readKeyPad())
      break;
  }
}
// ---------------------------------------------------------

void ReadList(const char* txt_buf = nullptr, const char* header = nullptr) {
  int lines = 0;
  int offset = 0;
  int index = 0;
  int row = 0;
  int key;
  char c = ' ';
  KeyMap *KP;

  display.clearDisplay();

  while (offset < strlen(txt_buf)-1) {
    while (c != '\n') {
      c = txt_buf[offset+index++];
      if (index < display.width())
        display.write(c),
        display.display();
    }
    offset += index;
    index = 0; c = ' ';
    if (++lines >= 8) break;
  }
  display.fillRect(0, (row*8), display.width()-1, 8, KS0108_INVERSE); display.display();

  while (1) {
    while ( !(KP = readKeyPad()) && !BOOTSEL ) yield();
    
    if (KP != NULL) { key = KP->key;
      if (key == 5) {
        break;
      }  
      else if (key == 2) {
        display.fillRect(0, (row*8), display.width()-1, 8, KS0108_INVERSE);
        row = (row > 0) ? row-1 : 0;
        display.fillRect(0, (row*8), display.width()-1, 8, KS0108_INVERSE); display.display();
      }
      else if (key == 8) {
        display.fillRect(0, (row*8), display.width()-1, 8, KS0108_INVERSE);
        if (row == lines-1) break;
        row = (row < lines-1) ? row+1 : lines-1;
        display.fillRect(0, (row*8), display.width()-1, 8, KS0108_INVERSE); display.display();
      }
      else {
        break;
      }
    }    
  }
  if ((row == lines-1) && (offset < strlen(txt_buf)-1))

  display.println("DONE"); display.display();
  delay(5000);
}
// ---------------------------------------------------------

void Reader(const char* txt_buf = nullptr, bool smallText = false) {
  uint16_t  last, index;
  uint8_t   key;
  bool      valid, curstate, multipage;
  KeyMap   *KP;

  if (txt_buf != NULL && strlen(txt_buf)) {
    setState(sys_status_prev.id);
    setState(READER);
    multipage = false;

    while (1) {
      curstate = false;
      index    = 0;
      
      display.clearDisplay();
      
      if (smallText)
        display.setFont(&Picopixel),
        display.setCursor(0,4);

      display.display();
      delay(100);
      
      while (index < strlen(txt_buf)-1) {
        BGL_set(ON);
        display.write(ascii2lcd(txt_buf[index++]));
        display.display();
        delay(10);
        
        if (display.getCursorY() > 56) {
          key       = 0;
          last      = index;
          valid     = false;
          multipage = true;
          
          while (!valid) {
            KP = readKeyPad();
            
            if (KP != NULL) {
              key = KP->key;
              
              if (key == 7 || key == 10 || key == 12) {
                valid = false;
                break;
              }
              else
                valid = true;
            }
            if ( !(millis() % 200) ) 
              GLCD_blinkCursor((curstate = !curstate), (20*6), 56, 25);
          }
          GLCD_blinkCursor(false, (20*6), 56, 25);
          
          if (KP != NULL) {
            if (!valid) break;
            
            display.pushUp(8);
            display.setCursor(0, 56);
            display.display();
          }
        }
      }
      display.setFont(NULL);

      GLCD_blinkCursor(true, (20*6), 56, 19);
      
      while ((readKeyPad() == NULL) && !BOOTSEL) yield();
      
      if (multipage) {
        GLCD_msgBox("\n  READ AGAIN?\n\n    PRESS 5", false);
        while (1) {
          KP = readKeyPad();
          if (KP != NULL) break;
        }
        if (KP->key != 5) break;
      }
      else
        break;
    }
    setState(sys_status_prev.id);   
  }
}
// ---------------------------------------------------------

void printFuncName(const char* id, uint8_t ypos = 56) {
  GLCD_overwrite(id, ypos, "'", "()'");
}

// ---------------------------------------------------------

void sysTick() {
  if ((bgl_timeout > 0) && ((millis()-bgl_timer)>bgl_timeout))
    BGL_set(OFF);

  if ((getState() == MENU_ON || getState() == HOME_ON) && (timeStatus() != timeNotSet)) {
    if ((millis()-clock_update) > 100) {
      clock_store();
    }
  }
}
// ---------------------------------------------------------

void Menu() {
  setState(MENU_ON);
  
  if (!menu_init) 
    menu_init = tree::Init(&menuTree, menu_init);

  BGL_set(ON);
  tree::Render();
}

// ---------------------------------------------------------

void printTopBar(const char* title) {
  display.fillRect(0,0,display.width()-1, 8, 0); display.display();
  
  if (tree::node_active != NULL)
    tree::showTitle((title == NULL) ? tree::node_active->label : title);
  
  clock_store();
}

// ---------------------------------------------------------

void awaitResponse() {
  setState(AWAIT_RESP);
  await_response = true;
  resp_timer = millis();
  GLCD_anim();
}
// ---------------------------------------------------------

uint8_t serial2hex() {
  uint8_t b;
  char temp[3];
  String s = Serial.readStringUntil('\n');

  strncpy(temp, s.substring(0, 2).c_str(), 2);

  for (byte n=0;n<2;n++) 
    temp[n] = (temp[n]>=97 && temp[n]<=122) ? temp[n]-32 : temp[n];
  
  b =  (temp[0]<='9') ? (temp[0]-'0') : (temp[0]-'A'+10); b *= 16;
  b += (temp[1]<='9') ? (temp[1]-'0') : (temp[1]-'A'+10);
  
  return b;
}
// ---------------------------------------------------------


bool danishKey(int16_t hex) {
  uint8_t hi = hex & 0xF0;
  uint8_t lo = hex & 0x0F;
  
  bool pressed = false;

  if (hi != 0xC0 && hi != 0xD0 && hi != 0xE0 && hi != 0xF0) return false;
  if (lo != 0x06 && lo != 0x05 && lo != 0x08) return false;

  if (hi == 0xC0 || hi == 0xD0) 
    Keyboard.press(KEY_LEFT_SHIFT);

  if (lo == 0x05) 
    Keyboard.press(KEY_A_RING),
    pressed = true;

  else if (lo == 0x06)
    Keyboard.press(KEY_ASH),
    pressed = true;

  else if (lo == 0x08)
    Keyboard.press(KEY_SLASHED_O),
    pressed = true;

  if (pressed) 
    Keyboard.releaseAll();

  return pressed;
}
// ---------------------------------------------------------

void specialKey(char c) {
  const char *special_chars = "@$[]{}<>æøåÆØÅ";
  char *ptr = strchr(special_chars, c);

  uint8_t x = (
    (c == '@') ? KEY_ATMARK :
    (c == '$') ? KEY_DOLLAR : 
    (c == '[') ? KEY_BRACE_LEFT :
    (c == ']') ? KEY_BRACE_RIGHT :
    (c == '{') ? KEY_CURLY_LEFT :
    (c == '}') ? KEY_CURLY_RIGHT :
    (c == '<') ? KEY_LT_SIGN :
    (c == '>') ? KEY_GT_SIGN : 0x00
  );

  if (ptr == NULL || x == 0x00) {
    Keyboard.write((uint8_t)c);
    return;
  }

  Keyboard.press(KEY_RIGHT_ALT);    
  Keyboard.press(x); delay(100);
  Keyboard.releaseAll();
}
// ---------------------------------------------------------

void sendCommand(bool confirm = true, bool await = true, uint8_t cmd_type = cmd_type_override) {
  KeyMap *KP;
  char *sep = strchr(buf_input, ' ');

  if (confirm) {
    GLCD_msgBox("\n  SEND COMMAND?\n\n    PRESS 5", false);
    while (1) {
      KP = readKeyPad();
      if (KP != NULL) break;
    }
  }
  
  if ((KP->key == 5) || !confirm) {
    if (strlen(buf_input) > 0) {
      if (cmd_type == CMD_HID) {
        for (int i=0; i<strlen(buf_input); i++) {
          if (!danishKey(buf_input[i])) 
            specialKey(buf_input[i]);
        }
        Keyboard.println();
      }
      else 
        Serial.println(buf_input);
    }
    if (await)
      awaitResponse();
    else 
      Home(true),
      delay(1000);
  }
  else
    Menu(); 
}
// ---------------------------------------------------------

void sketcher() {
  unsigned long t = millis();
  int16_t thisX   = display.width()/2;
  int16_t thisY   = display.height()/2;
  bool last_pixel = false;
  bool active     = true;
  bool printOut   = false;
  KeyMap *KP;
  
  //BGL_SET(ON);

  display.clearDisplay();
  display.drawRect(0, 0, display.width(), display.height(), 1);
  display.display();
    
  while (active) {
    KP = readKeyPad();
    
    if (KP != NULL) {
      uint8_t key = KP->key;

      //BGL_SET(ON);
      display.drawPixel(thisX, thisY, 0);

      if (last_pixel)
        display.drawPixel(thisX, thisY, 1);
                
      if (key == 2 || key == 4 || key == 6 || key == 8) {  
        if (key == 2) thisY = (thisY > 0) ? thisY-1 : thisY;
        if (key == 8) thisY = (thisY < display.height()) ? thisY+1 : thisY;
        if (key == 4) thisX = (thisX > 0) ? thisX-1 : thisX;
        if (key == 6) thisX = (thisX < display.width()) ? thisX+1 : thisX;
      }
      else if (key == 5) {
        display.drawPixel(thisX, thisY, KS0108_INVERSE);
      }
      last_pixel = display.getPixel(thisX, thisY);
      display.display();   
    }
    else if (KP == NULL) {
      if ( !(millis()%200) )
        display.drawPixel(thisX, thisY, KS0108_INVERSE),
        display.display();

      if (BOOTSEL) {
        t = millis();

        while (BOOTSEL) {
          uint32_t elapsed = ((millis()-t)/10);
          KP = readKeyPad();
          
          if (KP != NULL && (KP->key > 9)) {
            active = false;
            break;
          }
          if (elapsed <= 384) {  
            if (elapsed<128)
              display.drawLine(0, 0, elapsed, 0, 0);
            else if (elapsed >= 128 && elapsed < 192)
              display.drawLine(127, 0, 127, (elapsed-128), 0);
            else if (elapsed >= 192 && elapsed < 320)
              display.drawLine(127, 63, 127-(elapsed-192), 63, 0);
            else if (elapsed >= 320 && elapsed < 384) 
              display.drawLine(0, 63, 0, 63-(elapsed-320), 0);
            display.display();
            delay(4);
          }
          else if (elapsed > 384) {
            printOut = true;
            active = false;
          }
        }
        display.drawRect(0, 0, display.width(), display.height(), 1);
        display.display();       
      }
    }
  }
  
  if (printOut) {
    delay(1000);
    display.dumpPBM();
    delay(1000);
    t = millis();
    GLCD_msgBox("\n    SUCCESS!\n BUFFER PRINTED", false);
    while ((millis()-t)<3000) {
      KP = readKeyPad();
      if (KP != NULL)
        break;
    }
  }
}
// ---------------------------------------------------------

String sys_dateTimeStr(const int (&arr)[3], const char* sep = ":") {
  String val = "";
  int index = 0;

  for (const int &n : arr) 
    val = val
    + ((n<10) ? "0" : "") 
    + String(n) 
    + ((++index < 3) ? sep : "");

  return val;
}
// ---------------------------------------------------------

void cmd_webBrowser(const char* param = nullptr) {
  textInput("URL> ");

  while (1) {
    if (KP_T9_getChar(readKeyPad())) {
      GLCD_print("URL> ",0,1);
      GLCD_decodeStr(buf_input);
      display.write(curSym);
      KP_showIcon(true);
    }
    else if (BOOTSEL) { 
      while (BOOTSEL) yield();
      break;
    }
  }
  if (strlen(buf_input)) {
    Keyboard.print("cmd web ");
    sendCommand(false, true);
  }
}
// ---------------------------------------------------------

/*
void cmd_webBrowser(const char* param = nullptr) {
  if (param == NULL) {
    showWarning("\n    NO URL!");
    return;
  }
  buf_input[0] = '\0';
  strcpy(buf_input, "cmd web ");
  strcat(buf_input, param);
  sendCommand(false, true, CMD_HID);
}
// ---------------------------------------------------------
*/

void cmd_smsRead(const char* param = nullptr) {
  size_t digits = 4;
  int32_t num = -1;
  char str[digits+1] = {'\0'};
  
  KeyMap *KP = nullptr;

  if ((sysData[0].content != NULL) && (char2int(sysData[0].content[0]) > 0)) {
    GLCD_msgBox("\n  READ UNREAD?\n\n    PRESS 5", false);
    while (KP == NULL) KP = readKeyPad();
    if (KP->key == 5) num = 0;
  }
  if (num < 0) 
    num = numInput(digits, "Index> ");
  
  num = (num < 0) ? 0 : num;
  sprintf(str, "%d", num);

  buf_input[0] = '\0';
  strcpy(buf_input, "cmd sim read ");
  strcat(buf_input, str);
  sendCommand(false, true, CMD_HID);
}

// ---------------------------------------------------------

void cmd_smsSend(const char* param = nullptr) {
  char phone[16] = {'\0'};
  int key, index = 0;
  KeyMap *KP;

  textInput("Phone> ");
  while (1) {
    KP = readKeyPad();

    if (KP_T9_getChar(KP)) {
      GLCD_overflow();
      GLCD_print("Phone> ",0,1);
      GLCD_decodeStr(buf_input);
      display.write(curSym);
      KP_showIcon(true);
    }
    else if (BOOTSEL) {
      while (BOOTSEL) yield();
      break;
    }
  }
  
  if (strlen(buf_input) < 1) 
    return;

  for (int i=0; i<16; i++)
    phone[i] = buf_input[i];

  textInput("Message:\n> ");
  delay(500);

  while (1) {
    KP = readKeyPad();

    if (KP_T9_getChar(KP)) {
      GLCD_print("Message:\n> ",0,1);
      GLCD_decodeStr(buf_input);
      display.write(curSym);
      KP_showIcon(true);
    }
    else if (BOOTSEL) { 
      GLCD_msgBox("\n  SEND SMS?\n\n    PRESS 5", false);
      while (BOOTSEL) yield();
      while (KP == NULL) 
        KP = readKeyPad();
      break;
    }
  }
  if ((KP->key == 5) && strlen(buf_input)) {
    Keyboard.print("cmd sim send ");
    Keyboard.print(phone);
    Keyboard.print(" \x22");
    strcat(buf_input, "\x22");
    sendCommand(false, false);
  }
}  
// ---------------------------------------------------------

void cmd_smsPb(const char* param) {
  buf_input[0] = '\0';
  strcpy(buf_input, "cmd sim pb");
  sendCommand(false, true);
}

// ---------------------------------------------------------

void cmd_smsInbox(const char* param = nullptr) {
  buf_input[0] = '\0';
  strcpy(buf_input, "cmd sim inbox");
  sendCommand(false, true);
}
// ---------------------------------------------------------

void cmd_sendLcdBuf(const char* param = nullptr) {
  printFuncName(__func__);
  delay(500);
  //BGL_SET(ON);

  display.dumpPBM();

  display.setCursor(128-(5*6), 56);
  display.print(" DONE");
  display.display();
}
// ---------------------------------------------------------

void cmd_myFun(const char* param = nullptr) {
  printFuncName(__func__);
  delay(500);
  //BGL_SET(ON);

  for (int i = 0; i < 5; i++)
    digitalWrite(28, !digitalRead(28)),
    delay(500);
  digitalWrite(28, HIGH);

  display.setCursor(128-(5*6), 56);
  display.print(" DONE");
  display.display();
}
// ---------------------------------------------------------

void cmd_txtInput(const char* param = nullptr) {
  if (param == NULL)
    printFuncName(__func__),
    delay(500);
  
  //BGL_SET(ON);
  
  if (param != NULL) 
    cmd_type_override = (strcmp(param,"ser") == 0) ? CMD_SER : CMD_HID;
  textInput();
  cmd_type_override = CMD_HID;
}
// ---------------------------------------------------------

void cmd_sketch(const char* param = nullptr) {
  printFuncName(__func__);
  delay(500);
  //BGL_SET(ON);
  
  sketcher();
  Menu();
}
// ---------------------------------------------------------

void clock_show(const char* str, int16_t &cur) {
  int n = strlen(str);

  display.fillRect(111,1, 16, 7, 0);
  display.setFont(&Picopixel); 
  display.setCursor(109,6);

  for (int i = 9; i-->4;) {
    char c = str[(n-i)];
    if (c == ':') 
      cur = display.getCursorX();
    display.write(c);      
  }

  if ((timeStatus() != timeNotSet) && (getState() == HOME_ON)) {
    display.fillRect(21, 18, 40, 6, 0);
    display.setCursor(21, 22);
    display.print(strchr(dt_str, ' ')+1);
  }

  display.display();
  display.setFont(NULL);
  clock_update = millis();
}

// ---------------------------------------------------------

void clock_set(const char* raw) {
  unsigned long x = strtol(raw, 0, 10);
  setTime(x);
  timeSynced = true;
  clock_update = millis();
}
// ---------------------------------------------------------

void clock_store() {
  char *sep, prnt[3];

  if (timeStatus() != timeNotSet) { 
    dt_str[0] = homeScreen[0].content[0] = homeScreen[1].content[0] = '\0';

    for (int i=0; i<6; i++) {
      int val = (
        (i == 0) ? year()   :
        (i == 1) ? month()  :
        (i == 2) ? day()    : 
        (i == 3) ? hour()   :
        (i == 4) ? minute() :
        (i == 5) ? second() : 0
      );
      sprintf(prnt, "%02d", val);
      strcat( dt_str, prnt); strcat(dt_str, ((i<5)?((i>2)?":":((i<2)?"-":" ")):""));
    }
    sep = strchr(dt_str, ' '); *sep = '\0';
    strcpy(homeScreen[0].content, dt_str); *sep = ' '; sep++;
    strcpy(homeScreen[1].content, sep);
  }
  clock_show( (((getState() == MENU_ON || getState() == HOME_ON) && (timeStatus() != timeNotSet)) ? dt_str : "--:--:--"));
}
// ---------------------------------------------------------

time_t requestSync() {
  return 0; 
}
// ----------------------------------------

void printMoon(unsigned long day) {
  display.fillRect((display.width()/2)+1, 4, 63, 63, 1);
  display.fillRect((display.width()/2)+1, 4, 59, 59, 0);
  display.drawBitmap(
    (display.width()/2)+1, 4,
    ((day <= 1 ) ? moon_day1  :
     (day == 2 ) ? moon_day2  :
     (day == 3 ) ? moon_day3  :
     (day == 4 ) ? moon_day4  :
     (day == 5 ) ? moon_day5  :
     (day == 6 ) ? moon_day6  :
     (day == 7 ) ? moon_day7  :
     (day == 8 ) ? moon_day8  :
     (day == 9 ) ? moon_day9  :
     (day == 10) ? moon_day10 :
     (day == 11) ? moon_day11 :
     (day == 12) ? moon_day12 :
     (day == 13) ? moon_day13 :
     (day == 14) ? moon_day14 :
     (day == 15) ? moon_day15 :
     (day == 16) ? moon_day16 :
     (day == 17) ? moon_day17 :
     (day == 18) ? moon_day18 :
     (day == 19) ? moon_day19 :
     (day == 20) ? moon_day20 :
     (day == 21) ? moon_day21 :
     (day == 22) ? moon_day22 :
     (day == 23) ? moon_day23 :
     (day == 24) ? moon_day24 :
     (day == 25) ? moon_day25 :
     (day == 26) ? moon_day26 :
     (day == 27) ? moon_day27 :
     (day == 28) ? moon_day28 :
     (day >= 29) ? moon_day29 : moon_day14
    ),
    60, 60, 1);
  display.drawPixel(120, 55, 0);
  display.drawPixel(123 , 59, 0);
}
// ----------------------------------------

void checkUnread(bool disp = true) {
  if (char2int( sysData[0].content[0] ) > 0) {
    display.fillRect(display.width()-20, display.height()-11, 19, 10,  KS0108_OFF);
    display.drawRect(display.width()-20, display.height()-11, 20, 11, KS0108_ON);
    display.drawBitmap((display.width()-16), (display.height()-9), icon_letter, 9, 7, 1);
    display.setCursor(123, 60); display.write(sysData[0].content[0]);   
    if (disp)
      display.display();
  }
}
// ----------------------------------------

void printHomeItems(int16_t xPos, int16_t yPos) {
  display.setFont(&Picopixel); 
  display.cp437(true);

  display.fillRect(1, 10, 63, 53, 0);
  display.setCursor(xPos,yPos);

  for (int i=0; i<ArrLen(homeScreen); i++) {
    display.print(homeScreen[i].name);
    display.setCursor(21, yPos);

    if (strlen(homeScreen[i].content))
      display.print(homeScreen[i].content),
      display.print((i==5)?"'C":"");
    else
      display.print(" . . .");

    display.println();  
    display.setCursor(xPos, yPos = display.getCursorY());
  }

  display.drawLine(19, 8, 19, display.height()-1, KS0108_ON);

  checkUnread();
  display.display();
  display.setCursor(0,0);
  display.setFont(NULL);
}
// ---------------------------------------------------------

void homeArtwork(bool loaded = false) {
  char *sep, sun_pct[4] = {'\0'};
  int16_t x, sunPos;  

  sep = strchr(homeScreen[3].content, '%'); *sep = '\0';
  strcpy(sun_pct, homeScreen[3].content);   *sep = '%';
  x = atoi(sun_pct);

  // clear art frame
  display.fillRect(64, 9, (display.width()/2)-1, 53, 0);

  if (loaded) {
    display.drawBitmap(
      (display.width()-BMP_HLF_WIDTH), 
      (display.height()-BMP_HLF_HEIGHT)/2,
      ((x < 100) ? sunset_empty_bmp_64 : nightsky_bmp_64),
      BMP_HLF_WIDTH, BMP_HLF_HEIGHT, 1
    );    
    if (x < 100) {
      sunPos = (x < 50) ? 24 : (x/2);
      display.fillCircle(display.width()-32, sunPos-5, 10, 0);
      display.drawCircle(display.width()-32, sunPos-5, 10, 1);
      display.fillRect((display.width()-BMP_FOOTER_WIDTH), (display.height()-BMP_FOOTER_HEIGHT), display.height()-1, display.width()-1, 0);
      display.drawBitmap((display.width()-BMP_FOOTER_WIDTH), (display.height()-BMP_FOOTER_HEIGHT), sunset_sea_bmp_64, BMP_FOOTER_WIDTH, BMP_FOOTER_HEIGHT, 1);    
    }
    else {
      printMoon(moon_day);
    }
  }
  else {
    display.drawBitmap(
      (display.width()-BMP_HLF_WIDTH), 
      (display.height()-BMP_HLF_HEIGHT)/2,
      default_64,
      BMP_HLF_WIDTH, BMP_HLF_HEIGHT, 1
    );
  }

  display.drawLine(
    display.width()/2, 8, 
    display.width()/2, display.height()-1,
    KS0108_ON
  );

  display.display();
}
// ---------------------------------------------------------
